<?php
/**
 * Plugin Name: WooCommerce Price & Discount Manager Pro
 * Plugin URI: https://mostafanoor.ir/
 * Description: مدیریت حرفه‌ای قیمت و تخفیفات ووکامرس - افزایش و کاهش قیمت، گرد کردن قیمت، اعمال و حذف تخفیف به صورت دسته‌ای با سیستم لاگ پیشرفته و قابلیت بازگشت کامل. پشتیبانی کامل از محصولات ساده و متغیر.
 * Version: 3.1.1
 * Author: Mostafa Noor
 * Author URI: https://mostafanoor.ir/
 * Text Domain: wc-discount-manager
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.5
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

define('WCDM_VERSION', '3.2.0');
define('WCDM_BATCH_SIZE', 50); // تعداد محصولات در هر دسته
define('WCDM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WCDM_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WCDM_PLUGIN_BASENAME', plugin_basename(__FILE__));

// فعال‌سازی پلاگین
register_activation_hook(__FILE__, 'wcdm_activate_plugin');

function wcdm_activate_plugin() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    
    $log_table = $wpdb->prefix . 'wcdm_logs';
    $backup_table = $wpdb->prefix . 'wcdm_backups';
    
    $sql_logs = "CREATE TABLE $log_table (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        action_type varchar(50) NOT NULL,
        discount_type varchar(20) DEFAULT NULL,
        discount_value varchar(50) DEFAULT NULL,
        categories_affected text,
        brands_affected text,
        products_affected text,
        total_products int(11) NOT NULL DEFAULT 0,
        user_id bigint(20) unsigned NOT NULL,
        user_name varchar(100) NOT NULL,
        created_at datetime NOT NULL,
        scheduled_start datetime DEFAULT NULL,
        scheduled_end datetime DEFAULT NULL,
        ip_address varchar(45),
        notes text,
        PRIMARY KEY  (id),
        KEY action_type (action_type),
        KEY created_at (created_at)
    ) $charset_collate;";
    
    $sql_backups = "CREATE TABLE $backup_table (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        log_id bigint(20) unsigned NOT NULL,
        product_id bigint(20) unsigned NOT NULL,
        product_name varchar(255) NOT NULL,
        regular_price varchar(50),
        sale_price varchar(50),
        stock_status varchar(20) DEFAULT NULL,
        date_on_sale_from datetime DEFAULT NULL,
        date_on_sale_to datetime DEFAULT NULL,
        created_at datetime NOT NULL,
        PRIMARY KEY  (id),
        KEY log_id (log_id),
        KEY product_id (product_id)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_logs);
    dbDelta($sql_backups);
    
    // پاک کردن کامل کش‌ها موقع نصب/آپدیت
    if (function_exists('wc_delete_product_transients')) {
        // پاک کردن تمام transient های WooCommerce
        delete_transient('wc_products_onsale');
        delete_transient('wc_featured_products');
        wp_cache_flush();
    }
    
    // پاک کردن object cache
    wp_cache_flush();
    
    update_option('wcdm_version', WCDM_VERSION);
}

/**
 * کلاس اصلی پلاگین
 */
final class WC_Discount_Manager {
    
    private static $instance = null;
    private $log_table;
    private $backup_table;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        global $wpdb;
        $this->log_table = $wpdb->prefix . 'wcdm_logs';
        $this->backup_table = $wpdb->prefix . 'wcdm_backups';
        
        $this->init_hooks();
    }
    
    private function init_hooks() {
        add_action('admin_init', array($this, 'check_woocommerce'));
        add_action('admin_init', array($this, 'maybe_create_tables'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // فقط در صفحات ادمین پلاگین لود شود
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // AJAX handlers
        add_action('wp_ajax_wcdm_get_categories', array($this, 'ajax_get_categories'));
        add_action('wp_ajax_wcdm_get_brands', array($this, 'ajax_get_brands'));
        add_action('wp_ajax_wcdm_preview_products', array($this, 'ajax_preview_products'));
        add_action('wp_ajax_wcdm_remove_discounts', array($this, 'ajax_remove_discounts'));
        add_action('wp_ajax_wcdm_apply_discounts', array($this, 'ajax_apply_discounts'));
        add_action('wp_ajax_wcdm_get_logs', array($this, 'ajax_get_logs'));
        add_action('wp_ajax_wcdm_undo_action', array($this, 'ajax_undo_action'));
        add_action('wp_ajax_wcdm_clear_logs', array($this, 'ajax_clear_logs'));
        add_action('wp_ajax_wcdm_export_logs', array($this, 'ajax_export_logs'));
        add_action('wp_ajax_wcdm_search_products', array($this, 'ajax_search_products'));
        
        // Batch processing handlers
        add_action('wp_ajax_wcdm_start_batch', array($this, 'ajax_start_batch'));
        add_action('wp_ajax_wcdm_process_batch', array($this, 'ajax_process_batch'));
        add_action('wp_ajax_wcdm_undo_batch', array($this, 'ajax_undo_batch'));
    }
    
    public function maybe_create_tables() {
        global $wpdb;
        
        $logs_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->log_table}'") === $this->log_table;
        
        // Check if new columns exist in logs table
        $has_products_col = false;
        if ($logs_exists) {
            $col = $wpdb->get_results("SHOW COLUMNS FROM {$this->log_table} LIKE 'products_affected'");
            $has_products_col = !empty($col);
            
            $col_brands = $wpdb->get_results("SHOW COLUMNS FROM {$this->log_table} LIKE 'brands_affected'");
            $has_brands_col = !empty($col_brands);
        }

        $backups_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->backup_table}'") === $this->backup_table;
        
        // Check if new columns exist in backups table
        $has_stock_col = false;
        if ($backups_exists) {
            $col = $wpdb->get_results("SHOW COLUMNS FROM {$this->backup_table} LIKE 'stock_status'");
            $has_stock_col = !empty($col);
        }
        
        if (!$logs_exists || !$has_products_col || !$has_brands_col || !$backups_exists || !$has_stock_col) {
            wcdm_activate_plugin();
        }
    }
    
    public function check_woocommerce() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>';
                esc_html_e('پلاگین WooCommerce Discount Manager Pro نیاز به نصب و فعال‌سازی ووکامرس دارد.', 'wc-discount-manager');
                echo '</p></div>';
            });
        }
    }
    
    public function add_admin_menu() {
        add_menu_page(
            __('مدیریت قیمت محصولات', 'wc-discount-manager'),
            __('مدیریت قیمت محصولات', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager',
            array($this, 'render_admin_page'),
            'dashicons-tag',
            56
        );
        
        add_submenu_page(
            'wc-discount-manager',
            __('حذف تخفیفات', 'wc-discount-manager'),
            __('حذف تخفیفات', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager',
            array($this, 'render_admin_page')
        );
        
        add_submenu_page(
            'wc-discount-manager',
            __('افزودن تخفیف', 'wc-discount-manager'),
            __('افزودن تخفیف', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-add',
            array($this, 'render_add_discount_page')
        );

        add_submenu_page(
            'wc-discount-manager',
            __('افزایش قیمت', 'wc-discount-manager'),
            __('افزایش قیمت', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-increase',
            array($this, 'render_increase_page')
        );

        add_submenu_page(
            'wc-discount-manager',
            __('کاهش قیمت', 'wc-discount-manager'),
            __('کاهش قیمت', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-decrease',
            array($this, 'render_decrease_page')
        );

        add_submenu_page(
            'wc-discount-manager',
            __('گرد کردن قیمت', 'wc-discount-manager'),
            __('گرد کردن قیمت', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-rounding',
            array($this, 'render_rounding_page')
        );

        add_submenu_page(
            'wc-discount-manager',
            __('مدیریت موجودی', 'wc-discount-manager'),
            __('مدیریت موجودی', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-stock',
            array($this, 'render_stock_page')
        );
        
        add_submenu_page(
            'wc-discount-manager',
            __('گزارش فعالیت‌ها', 'wc-discount-manager'),
            __('گزارش فعالیت‌ها', 'wc-discount-manager'),
            'manage_woocommerce',
            'wc-discount-manager-logs',
            array($this, 'render_logs_page')
        );
    }
    
    public function enqueue_admin_assets($hook) {
        // فقط در صفحات پلاگین لود شود - مهم برای جلوگیری از تداخل
        if (strpos($hook, 'wc-discount-manager') === false) {
            return;
        }
        
        // Select2
        wp_enqueue_style('wcdm-select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css', array(), '4.1.0');
        wp_enqueue_script('wcdm-select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', array('jquery'), '4.1.0', true);
        
        // jQuery UI Datepicker
        wp_enqueue_script('jquery-ui-datepicker');
        wp_enqueue_style('wcdm-jquery-ui', 'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css', array(), '1.13.2');
        
        // Persian Datepicker (اختیاری)
        wp_enqueue_style('wcdm-admin', WCDM_PLUGIN_URL . 'assets/css/admin.css', array(), WCDM_VERSION);
        wp_enqueue_script('wcdm-admin', WCDM_PLUGIN_URL . 'assets/js/admin.js', array('jquery', 'wcdm-select2', 'jquery-ui-datepicker'), WCDM_VERSION, true);
        
        wp_localize_script('wcdm-admin', 'wcdm_vars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wcdm_nonce'),
            'batch_size' => WCDM_BATCH_SIZE,
            'strings' => array(
                'confirm_remove' => __('آیا از حذف تخفیفات اطمینان دارید؟', 'wc-discount-manager'),
                'confirm_apply' => __('آیا از اعمال تخفیفات اطمینان دارید؟', 'wc-discount-manager'),
                'processing' => __('در حال پردازش...', 'wc-discount-manager'),
                'success' => __('عملیات با موفقیت انجام شد.', 'wc-discount-manager'),
                'error' => __('خطا در انجام عملیات.', 'wc-discount-manager'),
                'select_category' => __('یک یا چند دسته‌بندی انتخاب کنید...', 'wc-discount-manager'),
                'select_brand' => __('یک یا چند برند انتخاب کنید...', 'wc-discount-manager'),
                'no_products' => __('محصولی یافت نشد.', 'wc-discount-manager'),
                'confirm_clear_logs' => __('آیا از پاک کردن تمام لاگ‌ها اطمینان دارید؟', 'wc-discount-manager'),
                'enter_discount' => __('لطفاً مقدار تخفیف را وارد کنید.', 'wc-discount-manager'),
                'batch_processing' => __('در حال پردازش دسته', 'wc-discount-manager'),
                'batch_complete' => __('پردازش تکمیل شد', 'wc-discount-manager'),
            )
        ));
    }
    
    public function render_admin_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/remove-page.php';
    }
    
    public function render_add_discount_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/add-page.php';
    }

    public function render_stock_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/stock-page.php';
    }

    public function render_increase_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/increase-page.php';
    }

    public function render_decrease_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/decrease-page.php';
    }

    public function render_rounding_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/rounding-page.php';
    }
    
    public function render_logs_page() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('شما دسترسی لازم را ندارید.', 'wc-discount-manager'));
        }
        include WCDM_PLUGIN_DIR . 'templates/logs-page.php';
    }
    
    // ========== AJAX Methods ==========
    
    public function ajax_search_products() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $term = isset($_POST['term']) ? sanitize_text_field($_POST['term']) : '';
        
        if (empty($term)) {
            wp_send_json_success(array());
        }
        
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => 20,
            's' => $term,
            'post_status' => 'publish',
            'fields' => 'ids'
        );
        
        $query = new WP_Query($args);
        $products = array();
        
        foreach ($query->posts as $id) {
            $product = wc_get_product($id);
            if ($product) {
                $products[] = array(
                    'id' => $id,
                    'text' => $product->get_name() . ' (#' . $id . ')'
                );
            }
        }
        
        wp_send_json_success($products);
    }

    public function ajax_get_categories() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $mode = isset($_POST['mode']) ? sanitize_text_field($_POST['mode']) : 'discounted';
        
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        ));
        
        $result = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $cat) {
                if ($mode === 'discounted') {
                    $count = $this->count_discounted_products_in_category($cat->term_id);
                    $label = $cat->name . ' (' . $count . ' تخفیف‌دار)';
                } else {
                    $count = $this->count_products_in_category($cat->term_id);
                    $label = $cat->name . ' (' . $count . ' محصول)';
                }
                
                $result[] = array(
                    'id' => $cat->term_id,
                    'text' => $label,
                    'count' => $count
                );
            }
        }
        
        wp_send_json_success($result);
    }
    
    private function count_products_in_category($category_id) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_status' => 'publish',
            'tax_query' => array(
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category_id
                )
            )
        );
        
        $query = new WP_Query($args);
        return $query->found_posts;
    }
    
    private function count_discounted_products_in_category($category_id) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_status' => 'publish',
            'tax_query' => array(
                array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $category_id
                )
            ),
            'meta_query' => array(
                array(
                    'key' => '_sale_price',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        $query = new WP_Query($args);
        return $query->found_posts;
    }
    
    public function ajax_get_brands() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        // Detect available brand taxonomy
        $taxonomy = 'product_brand'; // Default (WooCommerce Brands)
        $taxonomies = get_taxonomies();
        
        if (in_array('pwb-brand', $taxonomies)) {
            $taxonomy = 'pwb-brand'; // Perfect Brands
        } elseif (in_array('yith_product_brand', $taxonomies)) {
            $taxonomy = 'yith_product_brand'; // YITH
        } elseif (in_array('brand', $taxonomies)) {
            $taxonomy = 'brand'; // Generic
        }
        
        if (!taxonomy_exists($taxonomy)) {
             wp_send_json_error(__('هیچ طبقه بندی برندی یافت نشد', 'wc-discount-manager'));
        }
        
        $brands = get_terms(array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        ));
        
        $result = array();
        if (!is_wp_error($brands)) {
            foreach ($brands as $brand) {
                $count = $brand->count;
                $label = $brand->name . ' (' . $count . ' محصول)';
                
                $result[] = array(
                    'id' => $brand->term_id,
                    'text' => $label,
                    'count' => $count
                );
            }
        }
        
        wp_send_json_success(array('taxonomy' => $taxonomy, 'brands' => $result));
    }

    public function ajax_preview_products() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $categories = isset($_POST['categories']) ? array_map('intval', (array)$_POST['categories']) : array();
        $brands = isset($_POST['brands']) ? array_map('intval', (array)$_POST['brands']) : array();
        $product_ids = isset($_POST['product_ids']) ? array_map('intval', (array)$_POST['product_ids']) : array();
        $all_products = isset($_POST['all_products']) && $_POST['all_products'] === 'true';
        $mode = isset($_POST['mode']) ? sanitize_text_field($_POST['mode']) : 'discounted';
        $discount_type = isset($_POST['discount_type']) ? sanitize_text_field($_POST['discount_type']) : '';
        $discount_value = isset($_POST['discount_value']) ? floatval($_POST['discount_value']) : 0;
        $rounding_enabled = isset($_POST['rounding_enabled']) ? filter_var($_POST['rounding_enabled'], FILTER_VALIDATE_BOOLEAN) : false;
        $rounding_type = isset($_POST['rounding_type']) ? sanitize_text_field($_POST['rounding_type']) : 'up';
        $rounding_precision = isset($_POST['rounding_precision']) ? intval($_POST['rounding_precision']) : 1;
        
        if ($mode === 'discounted') {
            $products = $this->get_discounted_products($categories, $brands, $all_products);
        } else {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
        }
        
        if (empty($products)) {
            wp_send_json_success(array(
                'products' => array(),
                'total' => 0,
                'message' => __('محصولی یافت نشد.', 'wc-discount-manager')
            ));
        }
        
        $result = array();
        $total_count = 0;
        
        foreach ($products as $product_id) {
            $product = wc_get_product($product_id);
            if (!$product) continue;
            
            // بررسی اینکه آیا محصول متغیر است
            if ($product->is_type('variable')) {
                $variation_ids = $product->get_children();
                
                if (!empty($variation_ids)) {
                    // نمایش اطلاعات هر variation
                    foreach ($variation_ids as $variation_id) {
                        $variation = wc_get_product($variation_id);
                        if (!$variation) continue;
                        
                        $variation_data = $this->get_product_preview_data($variation, $mode, $discount_type, $discount_value, $rounding_enabled, $rounding_type, $rounding_precision);
                        if ($variation_data) {
                            // اضافه کردن نام parent به نام variation
                            $variation_data['name'] = $product->get_name() . ' - ' . $variation_data['name'];
                            $result[] = $variation_data;
                            $total_count++;
                        }
                    }
                }
            } else {
                // محصول ساده
                $product_data = $this->get_product_preview_data($product, $mode, $discount_type, $discount_value, $rounding_enabled, $rounding_type, $rounding_precision);
                if ($product_data) {
                    $result[] = $product_data;
                    $total_count++;
                }
            }
        }
        
        wp_send_json_success(array(
            'products' => $result,
            'total' => $total_count
        ));
    }
    
    // تابع کمکی برای دریافت اطلاعات پیش‌نمایش یک محصول
    private function get_product_preview_data($product, $mode, $discount_type, $discount_value, $rounding_enabled = false, $rounding_type = 'up', $rounding_precision = 1) {
        if (!$product) {
            return null;
        }
        
        $regular_price = floatval($product->get_regular_price());
        
        // برای افزایش، کاهش و افزودن تخفیف، محصولات بدون قیمت را نادیده بگیر
        if (($mode === 'increase' || $mode === 'decrease' || $mode === 'add') && $regular_price <= 0) {
            return null;
        }
        
        $current_sale_price = $product->get_sale_price();
        $stock_status = $product->get_stock_status();
        
        // محاسبه قیمت جدید برای پیش‌نمایش
        $new_sale_price = '';
        $new_regular_price = '';
        
        if ($mode === 'add' && $discount_value > 0 && $regular_price > 0) {
            if ($discount_type === 'percentage') {
                $new_sale_price = $regular_price - ($regular_price * $discount_value / 100);
            } else {
                $new_sale_price = $regular_price - $discount_value;
            }
            $new_sale_price = max(0, round($new_sale_price, 0));
        } elseif ($mode === 'increase' && $discount_value > 0 && $regular_price > 0) {
            if ($discount_type === 'percentage') {
                $new_regular_price = $regular_price + ($regular_price * $discount_value / 100);
            } else {
                $new_regular_price = $regular_price + $discount_value;
            }
            
            if ($rounding_enabled) {
                $new_regular_price = $this->apply_rounding($new_regular_price, $rounding_type, $rounding_precision);
            } else {
                $new_regular_price = max(0, round($new_regular_price, 0));
            }
        } elseif ($mode === 'decrease' && $discount_value > 0 && $regular_price > 0) {
            if ($discount_type === 'percentage') {
                $new_regular_price = $regular_price - ($regular_price * $discount_value / 100);
            } else {
                $new_regular_price = $regular_price - $discount_value;
            }
            
            if ($rounding_enabled) {
                $new_regular_price = $this->apply_rounding($new_regular_price, $rounding_type, $rounding_precision);
            } else {
                $new_regular_price = max(0, round($new_regular_price, 0));
            }
        } elseif ($mode === 'rounding') {
            // For rounding mode, we'll show both prices
            $rounding_type = isset($_POST['rounding_type']) ? sanitize_text_field($_POST['rounding_type']) : 'up';
            $price_type = isset($_POST['price_type']) ? sanitize_text_field($_POST['price_type']) : 'both';
            $precision = isset($_POST['rounding_precision']) ? intval($_POST['rounding_precision']) : 1;
            
            if ($price_type === 'both' || $price_type === 'regular') {
                $new_regular_price = $this->apply_rounding($regular_price, $rounding_type, $precision);
            }
            
            if (($price_type === 'both' || $price_type === 'sale') && !empty($current_sale_price)) {
                $new_sale_price = $this->apply_rounding(floatval($current_sale_price), $rounding_type, $precision);
            }
        }
        
        $current_discount = '';
        if (!empty($current_sale_price) && $regular_price > 0) {
            $current_discount = round((($regular_price - floatval($current_sale_price)) / $regular_price) * 100) . '%';
        }
        
        // دریافت نام محصول - برای variation نام attributes را نشان می‌دهد
        $product_name = $product->get_name();
        
        // دریافت دسته‌بندی - برای variation از parent استفاده می‌کنیم
        $product_id_for_categories = $product->is_type('variation') ? $product->get_parent_id() : $product->get_id();
        
        return array(
            'id' => $product->get_id(),
            'name' => $product_name,
            'regular_price' => wc_price($regular_price),
            'regular_price_raw' => $regular_price,
            'sale_price' => !empty($current_sale_price) ? wc_price($current_sale_price) : '-',
            'sale_price_raw' => $current_sale_price,
            'new_sale_price' => $new_sale_price ? wc_price($new_sale_price) : '',
            'new_sale_price_raw' => $new_sale_price,
            'new_regular_price' => $new_regular_price ? wc_price($new_regular_price) : '',
            'new_regular_price_raw' => $new_regular_price,
            'current_discount' => $current_discount,
            'stock_status' => $stock_status,
            'edit_link' => get_edit_post_link($product_id_for_categories, 'raw'),
            'categories' => wp_get_post_terms($product_id_for_categories, 'product_cat', array('fields' => 'names'))
        );
    }

    
    private function get_all_products($categories = array(), $brands = array(), $product_ids = array(), $all_products = false) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_status' => 'publish'
        );
        
        if (!$all_products) {
            if (!empty($product_ids)) {
                $args['post__in'] = $product_ids;
            } else {
                $tax_query = array('relation' => 'AND');
                
                if (!empty($categories)) {
                    $tax_query[] = array(
                        'taxonomy' => 'product_cat',
                        'field' => 'term_id',
                        'terms' => $categories
                    );
                }
                
                if (!empty($brands)) {
                     // Detect available brand taxonomy again or pass it
                    $taxonomy = 'product_brand';
                    $available_taxonomies = get_taxonomies();
                    if (in_array('pwb-brand', $available_taxonomies)) $taxonomy = 'pwb-brand';
                    elseif (in_array('yith_product_brand', $available_taxonomies)) $taxonomy = 'yith_product_brand';
                    elseif (in_array('brand', $available_taxonomies)) $taxonomy = 'brand';

                    $tax_query[] = array(
                        'taxonomy' => $taxonomy,
                        'field' => 'term_id',
                        'terms' => $brands
                    );
                }
                
                if (count($tax_query) > 1) {
                    $args['tax_query'] = $tax_query;
                }
            }
        }
        
        $query = new WP_Query($args);
        return $query->posts;
    }
    
    private function get_discounted_products($categories = array(), $brands = array(), $all_products = false) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_sale_price',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        if (!$all_products) {
            $tax_query = array('relation' => 'AND');
            
            if (!empty($categories)) {
                $tax_query[] = array(
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $categories
                );
            }
            
            if (!empty($brands)) {
                $taxonomy = 'product_brand';
                $available_taxonomies = get_taxonomies();
                if (in_array('pwb-brand', $available_taxonomies)) $taxonomy = 'pwb-brand';
                elseif (in_array('yith_product_brand', $available_taxonomies)) $taxonomy = 'yith_product_brand';
                elseif (in_array('brand', $available_taxonomies)) $taxonomy = 'brand';

                $tax_query[] = array(
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $brands
                );
            }
            
            if (count($tax_query) > 1) {
                $args['tax_query'] = $tax_query;
            }
        }
        
        $query = new WP_Query($args);
        return $query->posts;
    }
    
    // اعمال تخفیف - شروع پردازش دسته‌ای
    public function ajax_start_batch() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $categories = isset($_POST['categories']) ? array_map('intval', (array)$_POST['categories']) : array();
        $brands = isset($_POST['brands']) ? array_map('intval', (array)$_POST['brands']) : array();
        $product_ids = isset($_POST['product_ids']) ? array_map('intval', (array)$_POST['product_ids']) : array();
        $all_products = isset($_POST['all_products']) && $_POST['all_products'] === 'true';
        $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : 'apply';
        $discount_type = isset($_POST['discount_type']) ? sanitize_text_field($_POST['discount_type']) : 'percentage';
        $discount_value = isset($_POST['discount_value']) ? floatval($_POST['discount_value']) : 0;
        $schedule_start = isset($_POST['schedule_start']) ? sanitize_text_field($_POST['schedule_start']) : '';
        $schedule_end = isset($_POST['schedule_end']) ? sanitize_text_field($_POST['schedule_end']) : '';
        $notes = isset($_POST['notes']) ? sanitize_textarea_field($_POST['notes']) : '';
        $rounding_enabled = isset($_POST['rounding_enabled']) ? filter_var($_POST['rounding_enabled'], FILTER_VALIDATE_BOOLEAN) : false;
        $rounding_type = isset($_POST['rounding_type']) ? sanitize_text_field($_POST['rounding_type']) : 'up';
        $rounding_precision = isset($_POST['rounding_precision']) ? intval($_POST['rounding_precision']) : 1;
        $price_type = isset($_POST['price_type']) ? sanitize_text_field($_POST['price_type']) : 'both';
        
        // دریافت محصولات
        if ($action_type === 'remove') {
            $products = $this->get_discounted_products($categories, $brands, $all_products);
            $log_action = $all_products ? 'remove_all' : 'remove_category';
        } elseif ($action_type === 'set_outofstock') {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
            $log_action = 'set_outofstock';
        } elseif ($action_type === 'increase_price') {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
            $log_action = 'increase_price';
        } elseif ($action_type === 'decrease_price') {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
            $log_action = 'decrease_price';
        } elseif ($action_type === 'round_prices') {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
            $log_action = 'round_prices';
        } else {
            $products = $this->get_all_products($categories, $brands, $product_ids, $all_products);
            $log_action = 'apply_discount';
        }
        
        if (empty($products)) {
            wp_send_json_error(__('محصولی برای پردازش یافت نشد.', 'wc-discount-manager'));
        }
        
        $total_products = count($products);
        $total_batches = ceil($total_products / WCDM_BATCH_SIZE);
        
        // ایجاد لاگ
        $log_id = $this->create_log($log_action, $categories, $brands, $product_ids, $total_products, $notes, $discount_type, $discount_value, $schedule_start, $schedule_end);
        
        if (!$log_id) {
            wp_send_json_error(__('خطا در ایجاد لاگ', 'wc-discount-manager'));
        }
        
        // ذخیره اطلاعات پردازش در transient
        $batch_data = array(
            'log_id' => $log_id,
            'products' => $products,
            'action_type' => $action_type,
            'discount_type' => $discount_type,
            'discount_value' => $discount_value,
            'schedule_start' => $schedule_start,
            'schedule_end' => $schedule_end,
            'rounding_enabled' => $rounding_enabled,
            'rounding_type' => $rounding_type,
            'rounding_precision' => $rounding_precision,
            'price_type' => $price_type,
            'total_products' => $total_products,
            'processed' => 0,
            'errors' => 0,
            'current_batch' => 0,
            'total_batches' => $total_batches
        );
        
        set_transient('wcdm_batch_' . $log_id, $batch_data, HOUR_IN_SECONDS);
        
        wp_send_json_success(array(
            'log_id' => $log_id,
            'total_products' => $total_products,
            'total_batches' => $total_batches,
            'batch_size' => WCDM_BATCH_SIZE
        ));
    }
    
    // پردازش هر دسته
    public function ajax_process_batch() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $log_id = isset($_POST['log_id']) ? intval($_POST['log_id']) : 0;
        $batch_number = isset($_POST['batch_number']) ? intval($_POST['batch_number']) : 0;
        
        if (!$log_id) {
            wp_send_json_error(__('شناسه لاگ نامعتبر', 'wc-discount-manager'));
        }
        
        $batch_data = get_transient('wcdm_batch_' . $log_id);
        
        if (!$batch_data) {
            wp_send_json_error(__('اطلاعات پردازش یافت نشد', 'wc-discount-manager'));
        }
        
        $start = $batch_number * WCDM_BATCH_SIZE;
        $batch_products = array_slice($batch_data['products'], $start, WCDM_BATCH_SIZE);
        
        if (empty($batch_products)) {
            // پردازش تمام شده
            delete_transient('wcdm_batch_' . $log_id);
            $this->update_log_products($log_id, $batch_data['processed']);
            
            wp_send_json_success(array(
                'complete' => true,
                'processed' => $batch_data['processed'],
                'errors' => $batch_data['errors'],
                'message' => sprintf(
                    __('%d محصول با موفقیت پردازش شد.', 'wc-discount-manager'),
                    $batch_data['processed']
                )
            ));
        }
        
        $processed = 0;
        $errors = 0;
        
        // تبدیل تاریخ پایان
        $start_date = null;
        if (!empty($batch_data['schedule_start'])) {
            $start_date = date('Y-m-d 00:00:00', strtotime($batch_data['schedule_start']));
        }
        
        $end_date = null;
        if (!empty($batch_data['schedule_end'])) {
            $end_date = date('Y-m-d 23:59:59', strtotime($batch_data['schedule_end']));
        }
        
        foreach ($batch_products as $product_id) {
            // اطمینان از فرش بودن اطلاعات محصول
            clean_post_cache($product_id);
            $product = wc_get_product($product_id);
            
            if (!$product) {
                $errors++;
                continue;
            }
            
            // بررسی اینکه آیا محصول متغیر است
            $is_variable = $product->is_type('variable');
            
            if ($is_variable) {
                // پردازش تمام variations
                $variation_ids = $product->get_children();
                
                if (!empty($variation_ids)) {
                    foreach ($variation_ids as $variation_id) {
                        clean_post_cache($variation_id); // Clear cache for variation
                        $variation = wc_get_product($variation_id);
                        if (!$variation) {
                            $errors++;
                            continue;
                        }
                        
                        // پردازش variation
                        $result = $this->process_single_product(
                            $variation,
                            $log_id,
                            $batch_data,
                            $start_date,
                            $end_date
                        );
                        
                        if ($result) {
                            $processed++;
                        } else {
                            $errors++;
                        }
                    }
                    
                    // Sync Parent after variations update
                    WC_Product_Variable::sync($product_id);
                    wc_delete_product_transients($product_id);
                    clean_post_cache($product_id);
                } else {
                    // محصول متغیر بدون variation - خطا
                    $errors++;
                }
            } else {
                // محصول ساده - پردازش عادی
                $result = $this->process_single_product(
                    $product,
                    $log_id,
                    $batch_data,
                    $start_date,
                    $end_date
                );
                
                if ($result) {
                    $processed++;
                } else {
                    $errors++;
                }
            }
        }
        
        // به‌روزرسانی اطلاعات پردازش
        $batch_data['processed'] += $processed;
        $batch_data['errors'] += $errors;
        $batch_data['current_batch'] = $batch_number + 1;
        
        set_transient('wcdm_batch_' . $log_id, $batch_data, HOUR_IN_SECONDS);
        
        $progress = round(($batch_data['current_batch'] / $batch_data['total_batches']) * 100);
        
        wp_send_json_success(array(
            'complete' => false,
            'current_batch' => $batch_data['current_batch'],
            'total_batches' => $batch_data['total_batches'],
            'processed' => $batch_data['processed'],
            'errors' => $batch_data['errors'],
            'progress' => $progress
        ));
    }
    
    // تابع کمکی برای پردازش یک محصول (ساده یا variation)
    private function process_single_product($product, $log_id, $batch_data, $start_date, $end_date) {
        if (!$product) {
            return false;
        }
        
        // ذخیره بکاپ
        $this->save_backup($log_id, $product);
        
        try {
            // مقادیر اولیه (Current Values)
            $final_regular = $product->get_regular_price();
            $final_sale = $product->get_sale_price();
            
            // دریافت Timestamp برای تاریخ‌ها
            $current_date_from = $product->get_date_on_sale_from();
            $final_date_from = $current_date_from ? $current_date_from->getTimestamp() : '';
            
            $current_date_to = $product->get_date_on_sale_to();
            $final_date_to = $current_date_to ? $current_date_to->getTimestamp() : '';
            
            $final_stock_status = $product->get_stock_status();
            
            // --- اعمال تغییرات روی متغیرها ---
            
            if ($batch_data['action_type'] === 'remove') {
                // حذف تخفیف
                $final_sale = '';
                $final_date_from = '';
                $final_date_to = '';
                
            } elseif ($batch_data['action_type'] === 'set_outofstock') {
                // تغییر وضعیت به ناموجود
                $final_stock_status = 'outofstock';
                
            } elseif ($batch_data['action_type'] === 'increase_price') {
                // افزایش قیمت
                if ((float)$final_regular <= 0) return false;
                
                // محاسبه قیمت جدید عادی
                if ($batch_data['discount_type'] === 'percentage') {
                    $final_regular = (float)$final_regular + ((float)$final_regular * $batch_data['discount_value'] / 100);
                } else {
                    $final_regular = (float)$final_regular + $batch_data['discount_value'];
                }
                
                // تنظیمات گرد کردن
                $rounding_enabled = isset($batch_data['rounding_enabled']) && $batch_data['rounding_enabled'];
                $rounding_type = isset($batch_data['rounding_type']) ? $batch_data['rounding_type'] : 'up';
                $rounding_precision = isset($batch_data['rounding_precision']) ? intval($batch_data['rounding_precision']) : 1;
                
                if ($rounding_enabled) {
                    $final_regular = $this->apply_rounding($final_regular, $rounding_type, $rounding_precision);
                } else {
                    $final_regular = max(0, round($final_regular, 0));
                }
                
                // اگر قیمت تخفیفی دارد، آن را هم افزایش بده
                if (!empty($final_sale) && (float)$final_sale > 0) {
                    if ($batch_data['discount_type'] === 'percentage') {
                        $final_sale = (float)$final_sale + ((float)$final_sale * $batch_data['discount_value'] / 100);
                    } else {
                        $final_sale = (float)$final_sale + $batch_data['discount_value'];
                    }
                    
                    if ($rounding_enabled) {
                        $final_sale = $this->apply_rounding($final_sale, $rounding_type, $rounding_precision);
                    } else {
                        $final_sale = max(0, round($final_sale, 0));
                    }
                }
                
            } elseif ($batch_data['action_type'] === 'decrease_price') {
                // کاهش قیمت
                if ((float)$final_regular <= 0) return false;
                
                if ($batch_data['discount_type'] === 'percentage') {
                    $final_regular = (float)$final_regular - ((float)$final_regular * $batch_data['discount_value'] / 100);
                } else {
                    $final_regular = (float)$final_regular - $batch_data['discount_value'];
                }
                
                // تنظیمات گرد کردن
                $rounding_enabled = isset($batch_data['rounding_enabled']) && $batch_data['rounding_enabled'];
                $rounding_type = isset($batch_data['rounding_type']) ? $batch_data['rounding_type'] : 'up';
                $rounding_precision = isset($batch_data['rounding_precision']) ? intval($batch_data['rounding_precision']) : 1;
                
                if ($rounding_enabled) {
                    $final_regular = $this->apply_rounding($final_regular, $rounding_type, $rounding_precision);
                } else {
                    $final_regular = max(0, round($final_regular, 0));
                }
                
                // کاهش قیمت تخفیفی
                if (!empty($final_sale) && (float)$final_sale > 0) {
                    if ($batch_data['discount_type'] === 'percentage') {
                        $final_sale = (float)$final_sale - ((float)$final_sale * $batch_data['discount_value'] / 100);
                    } else {
                        $final_sale = (float)$final_sale - $batch_data['discount_value'];
                    }
                    
                    if ($rounding_enabled) {
                        $final_sale = $this->apply_rounding($final_sale, $rounding_type, $rounding_precision);
                    } else {
                        $final_sale = max(0, round($final_sale, 0));
                    }
                }
                
            } elseif ($batch_data['action_type'] === 'round_prices') {
                // گرد کردن قیمت
                $rounding_type = isset($batch_data['rounding_type']) ? $batch_data['rounding_type'] : 'up';
                $price_type = isset($batch_data['price_type']) ? $batch_data['price_type'] : 'both';
                $precision = isset($batch_data['rounding_precision']) ? intval($batch_data['rounding_precision']) : 1;
                
                if (($price_type === 'both' || $price_type === 'regular') && (float)$final_regular > 0) {
                    $final_regular = $this->apply_rounding((float)$final_regular, $rounding_type, $precision);
                }
                
                if (($price_type === 'both' || $price_type === 'sale') && !empty($final_sale) && (float)$final_sale > 0) {
                    $final_sale = $this->apply_rounding((float)$final_sale, $rounding_type, $precision);
                }
                
            } else {
                // اعمال تخفیف (Apply Discount)
                if ((float)$final_regular <= 0) return false;
                
                if ($batch_data['discount_type'] === 'percentage') {
                    $new_price = (float)$final_regular - ((float)$final_regular * $batch_data['discount_value'] / 100);
                } else {
                    $new_price = (float)$final_regular - $batch_data['discount_value'];
                }
                
                $final_sale = max(0, round($new_price, 0));
                
                if ($start_date) {
                    $final_date_from = strtotime($start_date);
                } else {
                    $final_date_from = '';
                }

                if ($end_date) {
                    $final_date_to = strtotime($end_date);
                } else {
                    $final_date_to = '';
                }
            }
            
            // --- فاز ذخیره‌سازی: Standard WooCommerce Save ---
            
            // 1. Regular Price
            $product->set_regular_price((string)$final_regular);
            
            // 2. Sale Price & Dates
            if ($final_sale !== '' && $final_sale !== null && (float)$final_sale >= 0) {
                $product->set_sale_price((string)$final_sale);
            } else {
                $product->set_sale_price('');
            }
            
            if ($final_date_from) {
                $product->set_date_on_sale_from($final_date_from);
            } else {
                $product->set_date_on_sale_from('');
            }
            
            if ($final_date_to) {
                $product->set_date_on_sale_to($final_date_to);
            } else {
                $product->set_date_on_sale_to('');
            }
            
            // 3. Stock Status
            $product->set_stock_status($final_stock_status);
            
            // 4. Save Product
            // Standard WooCommerce Save
            $product->save();
            
            // 5. DUAL-LAYER PERSISTENCE: FORCE DB UPDATE
            // We directly update the database to ensure no caching or hook interference can revert the price.
            global $wpdb;
            $postmeta_table = $wpdb->prefix . 'postmeta';
            $pid = $product->get_id();
            
            // Force Regular Price
            $wpdb->delete($postmeta_table, array('post_id' => $pid, 'meta_key' => '_regular_price'), array('%d', '%s'));
            $wpdb->insert($postmeta_table, array('post_id' => $pid, 'meta_key' => '_regular_price', 'meta_value' => (string)$final_regular), array('%d', '%s', '%s'));
            
            // Force Sale Price
            if ($final_sale !== '' && $final_sale !== null && (float)$final_sale >= 0) {
                 $wpdb->delete($postmeta_table, array('post_id' => $pid, 'meta_key' => '_sale_price'), array('%d', '%s'));
                 $wpdb->insert($postmeta_table, array('post_id' => $pid, 'meta_key' => '_sale_price', 'meta_value' => (string)$final_sale), array('%d', '%s', '%s'));
            } else {
                 $wpdb->delete($postmeta_table, array('post_id' => $pid, 'meta_key' => '_sale_price'), array('%d', '%s'));
            }
            
            // Calculate and Force Active Price (_price)
            $active_price = (string)$final_regular;
            // Determine if on sale
            $is_on_sale = false;
            if ($final_sale !== '' && $final_sale !== null) {
                 $sale_price_float = (float)$final_sale;
                 $regular_price_float = (float)$final_regular;
                 if ($sale_price_float < $regular_price_float && $sale_price_float >= 0) {
                     $now = current_time('timestamp');
                     $start = $final_date_from ? $final_date_from : 0;
                     $end = $final_date_to ? $final_date_to : 0;
                     
                     if ( (!$start || $start <= $now) && (!$end || $end >= $now) ) {
                         $active_price = (string)$final_sale;
                     }
                 }
            }
            
            $wpdb->delete($postmeta_table, array('post_id' => $pid, 'meta_key' => '_price'), array('%d', '%s'));
            $wpdb->insert($postmeta_table, array('post_id' => $pid, 'meta_key' => '_price', 'meta_value' => $active_price), array('%d', '%s', '%s'));

            // Force Dates
            if ($final_date_from) {
                update_post_meta($pid, '_sale_price_dates_from', $final_date_from);
            } else {
                delete_post_meta($pid, '_sale_price_dates_from');
            }
            if ($final_date_to) {
                update_post_meta($pid, '_sale_price_dates_to', $final_date_to);
             } else {
                delete_post_meta($pid, '_sale_price_dates_to');
            }
             
            // Force Stock
            update_post_meta($pid, '_stock_status', $final_stock_status);
            
            // Clear Caches Aggressively
            wc_delete_product_transients($pid);
            clean_post_cache($pid);
            wp_cache_delete($pid, 'post_meta');
            
            // Update Lookup Tables
            if (function_exists('wc_update_product_lookup_tables')) {
                wc_update_product_lookup_tables($pid);
            }
            
            // اگر محصول variation است، باید parent را هم سینک کنیم تا range قیمت آپدیت شود
            if ($product->is_type('variation')) {
                $parent_id = $product->get_parent_id();
                // We handle parent sync in the batch loop now for efficiency, 
                // but keeping a targeted clear here doesn't hurt.
                wc_delete_product_transients($parent_id);
                clean_post_cache($parent_id);
            }
            
            return true;
            
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * اعمال گرد کردن به قیمت با دقت مشخص
     */
    private function apply_rounding($price, $type, $precision = 1) {
        if ($precision <= 0) {
            $precision = 1;
        }
        
        if ($type === 'up') {
            return ceil($price / $precision) * $precision;
        } else {
            return floor($price / $precision) * $precision;
        }
    }

    
    // برگشت دسته‌ای
    public function ajax_undo_batch() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        $log_id = isset($_POST['log_id']) ? intval($_POST['log_id']) : 0;
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        
        if (!$log_id) {
            wp_send_json_error(__('شناسه لاگ نامعتبر است.', 'wc-discount-manager'));
        }
        
        global $wpdb;
        
        // دریافت تعداد کل بکاپ‌ها
        $total = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->backup_table} WHERE log_id = %d",
            $log_id
        ));
        
        if ($total == 0) {
            wp_send_json_error(__('اطلاعات بکاپ یافت نشد.', 'wc-discount-manager'));
        }
        
        // دریافت دسته فعلی
        $backups = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->backup_table} WHERE log_id = %d LIMIT %d OFFSET %d",
            $log_id,
            WCDM_BATCH_SIZE,
            $offset
        ));
        
        if (empty($backups)) {
            // پردازش تمام شده
            $wpdb->delete($this->backup_table, array('log_id' => $log_id), array('%d'));
            
            // ثبت لاگ برگشت
            $this->create_log('undo', array(), array(), array(), $total, sprintf(__('برگشت عملیات #%d', 'wc-discount-manager'), $log_id), '', '', '', '');
            
            wp_send_json_success(array(
                'complete' => true,
                'restored' => $total,
                'message' => sprintf(__('%d محصول به حالت قبل بازگردانده شد.', 'wc-discount-manager'), $total)
            ));
        }
        
        $restored = 0;
        
        foreach ($backups as $backup) {
            $product = wc_get_product($backup->product_id);
            if (!$product) continue;
            
            try {
                // بازگردانی قیمت عادی (regular price)
                if (!empty($backup->regular_price)) {
                    $product->set_regular_price($backup->regular_price);
                }
                
                // بازگردانی قیمت تخفیفی (sale price)
                if (!empty($backup->sale_price)) {
                    $product->set_sale_price($backup->sale_price);
                } else {
                    $product->set_sale_price('');
                }
                
                // بازگردانی تاریخ شروع تخفیف
                if (!empty($backup->date_on_sale_from)) {
                    $product->set_date_on_sale_from($backup->date_on_sale_from);
                } else {
                    $product->set_date_on_sale_from('');
                }
                
                // بازگردانی تاریخ پایان تخفیف
                if (!empty($backup->date_on_sale_to)) {
                    $product->set_date_on_sale_to($backup->date_on_sale_to);
                } else {
                    $product->set_date_on_sale_to('');
                }
                
                // بازگردانی وضعیت موجودی
                if (!empty($backup->stock_status)) {
                    $product->set_stock_status($backup->stock_status);
                }
                
                $product->save();
                
                // پاک کردن کامل کش محصول
                wc_delete_product_transients($backup->product_id);
                clean_post_cache($backup->product_id);
                
                // اگر variation بود، parent را هم clear کنیم
                if ($product->is_type('variation')) {
                    $parent_id = $product->get_parent_id();
                    WC_Product_Variable::sync($parent_id);
                    wc_delete_product_transients($parent_id);
                }
                
                $restored++;
            } catch (Exception $e) {
                continue;
            }
        }
        
        $new_offset = $offset + WCDM_BATCH_SIZE;
        $progress = round(($new_offset / $total) * 100);
        $progress = min(100, $progress);
        
        wp_send_json_success(array(
            'complete' => false,
            'offset' => $new_offset,
            'total' => $total,
            'restored' => $restored,
            'progress' => $progress
        ));
    }

    // اعمال تخفیف - قدیمی (برای سازگاری)
    public function ajax_apply_discounts() {
        // ریدایرکت به سیستم جدید
        $this->ajax_start_batch();
    }
    
    // حذف تخفیفات - قدیمی (برای سازگاری)
    public function ajax_remove_discounts() {
        $_POST['action_type'] = 'remove';
        $this->ajax_start_batch();
    }
    
    private function create_log($action_type, $categories, $brands, $product_ids, $total, $notes, $discount_type = '', $discount_value = '', $schedule_start = '', $schedule_end = '') {
        global $wpdb;
        
        $current_user = wp_get_current_user();
        
        $data = array(
            'action_type' => $action_type,
            'discount_type' => $discount_type,
            'discount_value' => $discount_value,
            'categories_affected' => !empty($categories) ? implode(',', $categories) : '',
            'brands_affected' => !empty($brands) ? implode(',', $brands) : '',
            'products_affected' => !empty($product_ids) ? implode(',', $product_ids) : '',
            'total_products' => $total,
            'user_id' => $current_user->ID,
            'user_name' => $current_user->display_name ? $current_user->display_name : $current_user->user_login,
            'ip_address' => $this->get_client_ip(),
            'notes' => $notes,
            'scheduled_start' => !empty($schedule_start) ? date('Y-m-d H:i:s', strtotime($schedule_start)) : null,
            'scheduled_end' => !empty($schedule_end) ? date('Y-m-d H:i:s', strtotime($schedule_end)) : null,
            'created_at' => current_time('mysql')
        );
        
        $result = $wpdb->insert($this->log_table, $data);
        
        if ($result === false) {
            return false;
        }
        
        return $wpdb->insert_id;
    }
    
    private function update_log_products($log_id, $count) {
        global $wpdb;
        $wpdb->update(
            $this->log_table,
            array('total_products' => $count),
            array('id' => $log_id),
            array('%d'),
            array('%d')
        );
    }
    
    private function save_backup($log_id, $product) {
        global $wpdb;
        
        $date_from = $product->get_date_on_sale_from();
        $date_to = $product->get_date_on_sale_to();
        
        $data = array(
            'log_id' => $log_id,
            'product_id' => $product->get_id(),
            'product_name' => $product->get_name(),
            'regular_price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'stock_status' => $product->get_stock_status(),
            'date_on_sale_from' => $date_from ? $date_from->format('Y-m-d H:i:s') : null,
            'date_on_sale_to' => $date_to ? $date_to->format('Y-m-d H:i:s') : null,
            'created_at' => current_time('mysql')
        );
        
        $wpdb->insert($this->backup_table, $data);
    }
    
    public function ajax_get_logs() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        global $wpdb;
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $per_page = 20;
        $offset = ($page - 1) * $per_page;
        
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->log_table}'") === $this->log_table;
        
        if (!$table_exists) {
            wp_send_json_success(array(
                'logs' => array(),
                'total' => 0,
                'pages' => 0,
                'current_page' => 1
            ));
        }
        
        $total = $wpdb->get_var("SELECT COUNT(*) FROM {$this->log_table}");
        
        $logs = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->log_table} ORDER BY created_at DESC LIMIT %d OFFSET %d",
            $per_page,
            $offset
        ));
        
        $result = array();
        if ($logs) {
            foreach ($logs as $log) {
                $categories = array();
                if (!empty($log->categories_affected)) {
                    $cat_ids = explode(',', $log->categories_affected);
                    foreach ($cat_ids as $cat_id) {
                        $term = get_term(intval($cat_id), 'product_cat');
                        if ($term && !is_wp_error($term)) {
                            $categories[] = $term->name;
                        }
                    }
                }
                
                $brands = array();
                if (!empty($log->brands_affected)) {
                    $brand_ids = explode(',', $log->brands_affected);
                    // Detect taxonomy here is risky but we can try common ones or just get_term if ID is unique enough (it is per taxonomy but get_term needs taxonomy or it searches all? No, it needs taxonomy usually but get_term($id) works if ID is known... wait, get_term requires taxonomy)
                    // We must assume taxonomy. Safe bet: try all common brand taxonomies for the ID.
                    
                    $available_taxonomies = get_taxonomies();
                    $brand_tax = 'product_brand';
                    if (in_array('pwb-brand', $available_taxonomies)) $brand_tax = 'pwb-brand';
                    elseif (in_array('yith_product_brand', $available_taxonomies)) $brand_tax = 'yith_product_brand';
                    elseif (in_array('brand', $available_taxonomies)) $brand_tax = 'brand';
                    
                    foreach ($brand_ids as $bid) {
                        $term = get_term(intval($bid), $brand_tax);
                        if ($term && !is_wp_error($term)) {
                            $brands[] = $term->name;
                        }
                    }
                }
                
                $action_labels = array(
                    'remove_category' => __('حذف از دسته‌بندی', 'wc-discount-manager'),
                    'apply_discount' => __('اعمال تخفیف', 'wc-discount-manager'),
                    'set_outofstock' => __('تغییر موجودی به ناموجود', 'wc-discount-manager'),
                    'increase_price' => __('افزایش قیمت', 'wc-discount-manager'),
                    'decrease_price' => __('کاهش قیمت', 'wc-discount-manager'),
                    'round_prices' => __('گرد کردن قیمت', 'wc-discount-manager'),
                    'undo' => __('برگشت', 'wc-discount-manager')
                );
                
                $discount_info = '';
                if ($log->action_type === 'apply_discount' && !empty($log->discount_value)) {
                    if ($log->discount_type === 'percentage') {
                        $discount_info = $log->discount_value . '%';
                    } else {
                        $discount_info = wc_price($log->discount_value);
                    }
                }

                
                $targets = '';
                if (!empty($log->categories_affected)) {
                    $targets = __('دسته‌ها: ', 'wc-discount-manager') . implode(', ', $categories);
                }
                
                if (!empty($log->brands_affected)) {
                    if ($targets) $targets .= ' | ';
                    $targets .= __('برندها: ', 'wc-discount-manager') . implode(', ', $brands);
                }
                
                if (empty($log->categories_affected) && empty($log->brands_affected)) {
                    if (!empty($log->products_affected)) {
                    $p_ids = explode(',', $log->products_affected);
                    $p_names = array();
                    $count = 0;
                    foreach ($p_ids as $pid) {
                        if ($count >= 5) {
                            $p_names[] = '...';
                            break;
                        }
                        $p = wc_get_product($pid);
                        if ($p) {
                            $p_names[] = $p->get_name();
                        } else {
                            $p_names[] = '#' . $pid;
                        }
                        $count++;
                    }
                    $targets = __('محصولات: ', 'wc-discount-manager') . implode('، ', $p_names);
                } elseif (empty($targets)) {
                    $targets = __('همه محصولات', 'wc-discount-manager');
                }
            }

                
                $schedule_info = '';
                if ($log->scheduled_start) {
                    $schedule_info .= __('از: ', 'wc-discount-manager') . date_i18n('Y/m/d', strtotime($log->scheduled_start));
                }
                if ($log->scheduled_end) {
                    if ($schedule_info) $schedule_info .= ' - ';
                    $schedule_info .= __('تا: ', 'wc-discount-manager') . date_i18n('Y/m/d', strtotime($log->scheduled_end));
                }
                
                $result[] = array(
                    'id' => $log->id,
                    'action_type' => isset($action_labels[$log->action_type]) ? $action_labels[$log->action_type] : $log->action_type,
                    'action_type_raw' => $log->action_type,
                    'discount_info' => $discount_info,
                    'total_products' => $log->total_products,
                    'categories' => $targets, // map to display column
                    'user_name' => $log->user_name,
                    'ip_address' => $log->ip_address,
                    'created_at' => date_i18n('Y/m/d H:i', strtotime($log->created_at)),
                    'scheduled_end' => $schedule_info, // reuse validation column
                    'notes' => $log->notes,
                    'can_undo' => $this->can_undo($log->id)
                );
            }
        }
        
        wp_send_json_success(array(
            'logs' => $result,
            'total' => intval($total),
            'pages' => ceil($total / $per_page),
            'current_page' => $page
        ));
    }
    
    private function can_undo($log_id) {
        global $wpdb;
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->backup_table} WHERE log_id = %d",
            $log_id
        ));
        return $count > 0;
    }
    
    public function ajax_undo_action() {
        // شروع برگشت دسته‌ای
        $_POST['offset'] = 0;
        $this->ajax_undo_batch();
    }
    
    public function ajax_clear_logs() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        global $wpdb;
        
        $wpdb->query("TRUNCATE TABLE {$this->backup_table}");
        $wpdb->query("TRUNCATE TABLE {$this->log_table}");
        
        wp_send_json_success(array(
            'message' => __('تمام لاگ‌ها پاک شد.', 'wc-discount-manager')
        ));
    }
    
    public function ajax_export_logs() {
        check_ajax_referer('wcdm_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('دسترسی غیرمجاز', 'wc-discount-manager'));
        }
        
        global $wpdb;
        
        $logs = $wpdb->get_results("SELECT * FROM {$this->log_table} ORDER BY created_at DESC");
        
        $csv_data = array();
        $csv_data[] = array('شناسه', 'نوع عملیات', 'مقدار تخفیف', 'تعداد', 'دسته‌بندی‌ها', 'کاربر', 'تاریخ', 'پایان تخفیف', 'یادداشت');
        
        if ($logs) {
            foreach ($logs as $log) {
                $categories = array();
                if (!empty($log->categories_affected)) {
                    $cat_ids = explode(',', $log->categories_affected);
                    foreach ($cat_ids as $cat_id) {
                        $term = get_term(intval($cat_id), 'product_cat');
                        if ($term && !is_wp_error($term)) {
                            $categories[] = $term->name;
                        }
                    }
                }
                
                $csv_data[] = array(
                    $log->id,
                    $log->action_type,
                    $log->discount_value ? $log->discount_value . ($log->discount_type === 'percentage' ? '%' : ' تومان') : '-',
                    $log->total_products,
                    implode(' | ', $categories),
                    $log->user_name,
                    $log->created_at,
                    $log->scheduled_end ? $log->scheduled_end : '-',
                    $log->notes
                );
            }
        }
        
        wp_send_json_success(array('data' => $csv_data));
    }
    
    private function get_client_ip() {
        $ip = '';
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_CLIENT_IP']));
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR']));
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }
        return $ip;
    }
}

function wcdm_init() {
    return WC_Discount_Manager::get_instance();
}

add_action('plugins_loaded', 'wcdm_init');
