/**
 * WooCommerce Discount Manager Pro - Admin Scripts
 * Version: 2.1.0 - Batch Processing Support
 */

(function ($) {
    'use strict';

    const WCDM = {

        init: function () {
            this.bindEvents();
            this.initSelect2();
            this.initDatepicker();
            this.loadCategories();
            this.loadRecentLogs();

            if ($('#wcdm-logs-table').length) {
                this.loadLogs(1);
            }
        },

        bindEvents: function () {
            // Remove page events
            $('input[name="remove_action_type"]').on('change', this.handleRemoveTypeChange.bind(this));
            $('#wcdm-remove-preview-btn').on('click', this.handleRemovePreview.bind(this));
            $('#wcdm-remove-btn').on('click', this.handleRemove.bind(this));

            // Add page events
            $('input[name="add_action_type"]').on('change', this.handleAddTypeChange.bind(this));
            $('input[name="discount_type"]').on('change', this.handleDiscountTypeChange.bind(this));
            $('#wcdm-add-preview-btn').on('click', this.handleAddPreview.bind(this));
            $('#wcdm-apply-btn').on('click', this.handleApply.bind(this));
            $('#wcdm-discount-value').on('input', this.validateDiscountValue.bind(this));

            // Stock page events
            $('input[name="stock_action_type"]').on('change', this.handleStockTypeChange.bind(this));
            $('#wcdm-stock-preview-btn').on('click', this.handleStockPreview.bind(this));
            $('#wcdm-stock-apply-btn').on('click', this.handleStockApply.bind(this));

            // Increase page events
            $('input[name="increase_action_type"]').on('change', this.handleIncreaseTypeChange.bind(this));
            $('input[name="increase_type"]').on('change', this.handleIncreaseValueTypeChange.bind(this));
            $('#wcdm-increase-preview-btn').on('click', this.handleIncreasePreview.bind(this));
            $('#wcdm-increase-apply-btn').on('click', this.handleIncreaseApply.bind(this));
            $('#wcdm-increase-value').on('input', this.validateIncreaseValue.bind(this));
            $('#wcdm-increase-rounding-enable').on('change', function () {
                if ($(this).is(':checked')) {
                    $('#wcdm-increase-rounding-options').slideDown(300);
                } else {
                    $('#wcdm-increase-rounding-options').slideUp(300);
                }
            });
            $('input[name="increase_rounding_type"]').on('change', function () {
                $('.wcdm-discount-type-option').removeClass('active');
                $('input[name="increase_rounding_type"]:checked').closest('.wcdm-discount-type-option').addClass('active');
            });

            // Decrease page events
            $('input[name="decrease_action_type"]').on('change', this.handleDecreaseTypeChange.bind(this));
            $('input[name="decrease_type"]').on('change', this.handleDecreaseValueTypeChange.bind(this));
            $('#wcdm-decrease-preview-btn').on('click', this.handleDecreasePreview.bind(this));
            $('#wcdm-decrease-apply-btn').on('click', this.handleDecreaseApply.bind(this));
            $('#wcdm-decrease-value').on('input', this.validateDecreaseValue.bind(this));
            $('#wcdm-decrease-rounding-enable').on('change', function () {
                if ($(this).is(':checked')) {
                    $('#wcdm-decrease-rounding-options').slideDown(300);
                } else {
                    $('#wcdm-decrease-rounding-options').slideUp(300);
                }
            });

            // Rounding page events
            $('input[name="rounding_action_type"]').on('change', this.handleRoundingTypeChange.bind(this));
            $('input[name="rounding_direction"]').on('change', this.handleRoundingDirectionChange.bind(this));
            $('input[name="rounding_price_type"]').on('change', this.handleRoundingPriceTypeChange.bind(this));
            $('#wcdm-rounding-preview-btn').on('click', this.handleRoundingPreview.bind(this));
            $('#wcdm-rounding-apply-btn').on('click', this.handleRoundingApply.bind(this));

            // Logs page events
            $(document).on('click', '.wcdm-undo-btn', this.handleUndo.bind(this));
            $('#wcdm-clear-logs').on('click', this.handleClearLogs.bind(this));
            $('#wcdm-export-logs').on('click', this.handleExportLogs.bind(this));
            $(document).on('click', '.wcdm-pagination button', this.handlePagination.bind(this));

            // Modal events
            $(document).on('click', '.wcdm-modal-close, .wcdm-modal-close-btn', this.closeModal);
            $(document).on('click', '.wcdm-modal', function (e) {
                if ($(e.target).hasClass('wcdm-modal')) {
                    WCDM.closeModal();
                }
            });
            $(document).on('keyup', function (e) {
                if (e.key === 'Escape') {
                    WCDM.closeModal();
                }
            });
        },

        initSelect2: function () {
            $('.wcdm-select2').each(function () {
                const $this = $(this);
                const type = $this.data('type');
                const placeholder = type === 'brand' ? wcdm_vars.strings.select_brand : wcdm_vars.strings.select_category;

                $this.select2({
                    placeholder: placeholder,
                    allowClear: true,
                    dir: 'rtl',
                    width: '100%'
                });
            });

            // Product Search Select2
            $('.wcdm-select2-products').select2({
                placeholder: 'جستجوی محصول...',
                allowClear: true,
                dir: 'rtl',
                width: '100%',
                ajax: {
                    url: wcdm_vars.ajax_url,
                    dataType: 'json',
                    delay: 250,
                    data: function (params) {
                        return {
                            action: 'wcdm_search_products',
                            nonce: wcdm_vars.nonce,
                            term: params.term
                        };
                    },
                    processResults: function (data) {
                        return {
                            results: data.data || []
                        };
                    },
                    cache: true
                },
                minimumInputLength: 3
            });
        },

        initDatepicker: function () {
            if ($('.wcdm-datepicker').length) {
                $('.wcdm-datepicker').datepicker({
                    dateFormat: 'yy-mm-dd',
                    minDate: 0,
                    isRTL: true,
                    changeMonth: true,
                    changeYear: true
                });
            }
        },

        loadCategories: function () {
            $('.wcdm-select2').each(function () {
                const $select = $(this);
                const type = $select.data('type');
                const mode = $select.data('mode') || 'discounted';

                const action = type === 'brand' ? 'wcdm_get_brands' : 'wcdm_get_categories';

                $.ajax({
                    url: wcdm_vars.ajax_url,
                    type: 'POST',
                    data: {
                        action: action,
                        nonce: wcdm_vars.nonce,
                        mode: mode
                    },
                    success: function (response) {
                        if (response.success) {
                            $select.empty();
                            const items = type === 'brand' ? response.data.brands : response.data;
                            if (items) {
                                items.forEach(function (item) {
                                    $select.append(new Option(item.text, item.id, false, false));
                                });
                            }
                            $select.trigger('change');
                        }
                    }
                });
            });
        },

        loadRecentLogs: function () {
            if (!$('#wcdm-recent-logs').length) return;

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_get_logs',
                    nonce: wcdm_vars.nonce,
                    page: 1
                },
                success: function (response) {
                    if (response.success && response.data.logs.length) {
                        let html = '';
                        const logs = response.data.logs.slice(0, 5);

                        logs.forEach(function (log) {
                            html += '<div class="wcdm-recent-log-item wcdm-fade-in">';
                            html += '<span class="wcdm-recent-log-type ' + log.action_type_raw + '">' + log.action_type + '</span>';
                            html += '<div class="wcdm-recent-log-info">';
                            html += '<span>' + log.total_products + ' محصول</span>';
                            html += '<span>' + log.created_at + '</span>';
                            html += '</div></div>';
                        });

                        $('#wcdm-recent-logs').html(html);
                    } else {
                        $('#wcdm-recent-logs').html('<p class="wcdm-empty-state">هیچ فعالیتی ثبت نشده است.</p>');
                    }
                }
            });
        },

        // Remove Page Handlers
        handleRemoveTypeChange: function () {
            const type = $('input[name="remove_action_type"]:checked').val();

            $('#wcdm-remove-category-card').hide();
            $('#wcdm-remove-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-remove-category-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-remove-brand-card').slideDown(300);
            }

            $('#wcdm-remove-preview-card').hide();
            $('#wcdm-remove-btn').prop('disabled', true);
        },

        handleRemovePreview: function () {
            const type = $('input[name="remove_action_type"]:checked').val();
            const categories = $('#wcdm-remove-categories').val() || [];

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: $('#wcdm-remove-categories').val() || [],
                    brands: $('#wcdm-remove-brands').val() || [],
                    all_products: type === 'all',
                    mode: 'discounted'
                },
                success: function (response) {
                    WCDM.hideProgress();
                    if (response.success) {
                        WCDM.renderRemovePreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderRemovePreview: function (data) {
            const $card = $('#wcdm-remove-preview-card');
            const $tbody = $('#wcdm-remove-preview-table tbody');
            const $count = $('#wcdm-remove-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="5" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-remove-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + product.regular_price + '</td>' +
                        '<td>' + product.sale_price + '</td>' +
                        '<td><span class="wcdm-discount-badge">' + product.current_discount + '</span></td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="5" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-remove-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleRemove: function () {
            if (!confirm(wcdm_vars.strings.confirm_remove)) return;

            const type = $('input[name="remove_action_type"]:checked').val();
            const categories = $('#wcdm-remove-categories').val() || [];
            const notes = $('#wcdm-remove-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-remove-brands').val() || [],
                    all_products: type === 'all',
                    action_type: 'remove',
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'remove');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        // Add Page Handlers
        handleAddTypeChange: function () {
            const type = $('input[name="add_action_type"]:checked').val();

            $('#wcdm-add-category-card').hide();
            $('#wcdm-add-product-card').hide();
            $('#wcdm-add-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-add-category-card').slideDown(300);
            } else if (type === 'product') {
                $('#wcdm-add-product-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-add-brand-card').slideDown(300);
            }

            $('#wcdm-add-preview-card').hide();
            $('#wcdm-apply-btn').prop('disabled', true);
        },

        handleDiscountTypeChange: function () {
            const type = $('input[name="discount_type"]:checked').val();

            $('.wcdm-discount-type-option').removeClass('active');
            $('input[name="discount_type"]:checked').closest('.wcdm-discount-type-option').addClass('active');

            if (type === 'percentage') {
                $('#wcdm-discount-suffix').text('%');
            } else {
                $('#wcdm-discount-suffix').text('تومان');
            }

            this.validateDiscountValue();
        },

        validateDiscountValue: function () {
            const type = $('input[name="discount_type"]:checked').val();
            let value = parseFloat($('#wcdm-discount-value').val()) || 0;

            if (type === 'percentage' && value > 100) {
                $('#wcdm-discount-value').val(100);
            }
        },

        handleAddPreview: function () {
            const type = $('input[name="add_action_type"]:checked').val();
            const categories = $('#wcdm-add-categories').val() || [];
            const productIds = $('#wcdm-add-products').val() || [];
            const discountType = $('input[name="discount_type"]:checked').val();
            const discountValue = parseFloat($('#wcdm-discount-value').val()) || 0;

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            if (type === 'product' && productIds.length === 0) {
                this.showToast('لطفاً حداقل یک محصول انتخاب کنید.', 'warning');
                return;
            }

            if (discountValue <= 0) {
                this.showToast(wcdm_vars.strings.enter_discount, 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-add-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    mode: 'add',
                    discount_type: discountType,
                    discount_value: discountValue
                },
                success: function (response) {
                    WCDM.hideProgress();
                    if (response.success) {
                        WCDM.renderAddPreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderAddPreview: function (data) {
            const $card = $('#wcdm-add-preview-card');
            const $tbody = $('#wcdm-add-preview-table tbody');
            const $count = $('#wcdm-add-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="5" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-apply-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + product.regular_price + '</td>' +
                        '<td>' + product.sale_price + '</td>' +
                        '<td><span class="wcdm-new-price">' + product.new_sale_price + '</span></td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="5" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-apply-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleApply: function () {
            if (!confirm(wcdm_vars.strings.confirm_apply)) return;

            const type = $('input[name="add_action_type"]:checked').val();
            const categories = $('#wcdm-add-categories').val() || [];
            const productIds = $('#wcdm-add-products').val() || [];
            const discountType = $('input[name="discount_type"]:checked').val();
            const discountValue = parseFloat($('#wcdm-discount-value').val()) || 0;
            const scheduleStart = $('#wcdm-schedule-start').val();
            const scheduleEnd = $('#wcdm-schedule-end').val();
            const notes = $('#wcdm-add-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-add-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    action_type: 'apply',
                    discount_type: discountType,
                    discount_value: discountValue,
                    schedule_start: scheduleStart,
                    schedule_end: scheduleEnd,
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'apply');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        // Stock Page Handlers
        handleStockTypeChange: function () {
            const type = $('input[name="stock_action_type"]:checked').val();

            $('#wcdm-stock-category-card').hide();
            $('#wcdm-stock-product-card').hide();
            $('#wcdm-stock-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-stock-category-card').slideDown(300);
            } else if (type === 'product') {
                $('#wcdm-stock-product-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-stock-brand-card').slideDown(300);
            }

            $('#wcdm-stock-preview-card').hide();
            $('#wcdm-stock-apply-btn').prop('disabled', true);
        },

        handleStockPreview: function () {
            const type = $('input[name="stock_action_type"]:checked').val();
            const categories = $('#wcdm-stock-categories').val() || [];
            const productIds = $('#wcdm-stock-products').val() || [];

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            if (type === 'product' && productIds.length === 0) {
                this.showToast('لطفاً حداقل یک محصول انتخاب کنید.', 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-stock-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    mode: 'stock'
                },
                success: function (response) {
                    WCDM.hideProgress();
                    if (response.success) {
                        WCDM.renderStockPreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderStockPreview: function (data) {
            const $card = $('#wcdm-stock-preview-card');
            const $tbody = $('#wcdm-stock-preview-table tbody');
            const $count = $('#wcdm-stock-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="5" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-stock-apply-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + (product.stock_status === 'instock' ? '<span class="wcdm-badge wcdm-badge-success">موجود</span>' : '<span class="wcdm-badge wcdm-badge-danger">ناموجود</span>') + '</td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="5" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-stock-apply-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleStockApply: function () {
            if (!confirm('آیا از تغییر وضعیت موجودی محصولات اطمینان دارید؟')) return;

            const type = $('input[name="stock_action_type"]:checked').val();
            const categories = $('#wcdm-stock-categories').val() || [];
            const productIds = $('#wcdm-stock-products').val() || [];
            const notes = $('#wcdm-stock-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-stock-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    action_type: 'set_outofstock',
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'stock');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        processBatch: function (logId, batchNumber, totalBatches, totalProducts, actionType) {
            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_process_batch',
                    nonce: wcdm_vars.nonce,
                    log_id: logId,
                    batch_number: batchNumber
                },
                success: function (response) {
                    if (response.success) {
                        if (response.data.complete) {
                            WCDM.hideProgress();
                            WCDM.showResult('عملیات موفق', response.data.message);

                            if (actionType === 'remove') {
                                $('#wcdm-remove-preview-card').hide();
                                $('#wcdm-remove-btn').prop('disabled', true);
                                $('#wcdm-remove-notes').val('');
                            } else if (actionType === 'stock') {
                                $('#wcdm-stock-preview-card').hide();
                                $('#wcdm-stock-apply-btn').prop('disabled', true);
                                $('#wcdm-stock-notes').val('');
                            } else if (actionType === 'increase') {
                                $('#wcdm-increase-preview-card').hide();
                                $('#wcdm-increase-apply-btn').prop('disabled', true);
                                $('#wcdm-increase-value').val('');
                                $('#wcdm-increase-notes').val('');
                            } else if (actionType === 'decrease') {
                                $('#wcdm-decrease-preview-card').hide();
                                $('#wcdm-decrease-apply-btn').prop('disabled', true);
                                $('#wcdm-decrease-value').val('');
                                $('#wcdm-decrease-notes').val('');
                            } else if (actionType === 'rounding') {
                                $('#wcdm-rounding-preview-card').hide();
                                $('#wcdm-rounding-apply-btn').prop('disabled', true);
                                $('#wcdm-rounding-notes').val('');
                            } else {
                                $('#wcdm-add-preview-card').hide();
                                $('#wcdm-apply-btn').prop('disabled', true);
                                $('#wcdm-discount-value').val('');
                                $('#wcdm-schedule-end').val('');
                                $('#wcdm-add-notes').val('');
                            }

                            WCDM.loadCategories();
                            WCDM.loadRecentLogs();

                            if (actionType === 'remove') {
                                const $stat = $('.wcdm-header-stats .wcdm-stat-number');
                                const current = parseInt($stat.text()) || 0;
                                $stat.text(Math.max(0, current - response.data.processed));
                            }
                        } else {
                            WCDM.showBatchProgress(
                                response.data.progress,
                                response.data.current_batch,
                                totalBatches,
                                'پردازش شده: ' + response.data.processed + ' از ' + totalProducts
                            );

                            setTimeout(function () {
                                WCDM.processBatch(logId, response.data.current_batch, totalBatches, totalProducts, actionType);
                            }, 100);
                        }
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        // Logs Page
        loadLogs: function (page) {
            this.currentPage = page;

            const $tbody = $('#wcdm-logs-body');
            $tbody.html('<tr><td colspan="9" class="wcdm-loading-row">' +
                '<span class="spinner is-active"></span> در حال بارگذاری...</td></tr>');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_get_logs',
                    nonce: wcdm_vars.nonce,
                    page: page
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.renderLogs(response.data);
                    }
                }
            });
        },

        renderLogs: function (data) {
            const $tbody = $('#wcdm-logs-body');
            const $pagination = $('#wcdm-pagination');

            $tbody.empty();

            if (data.logs.length === 0) {
                $tbody.html('<tr><td colspan="9" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-clipboard"></span>' +
                    '<h3>هیچ لاگی یافت نشد</h3></td></tr>');
                $pagination.empty();
                $('#wcdm-stat-total').text('0');
                $('#wcdm-stat-products').text('0');
                $('#wcdm-stat-undos').text('0');
                return;
            }

            let totalProducts = 0;
            let undoCount = 0;

            data.logs.forEach(function (log, index) {
                totalProducts += parseInt(log.total_products);
                if (log.can_undo) undoCount++;

                $tbody.append(
                    '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                    '<td>#' + log.id + '</td>' +
                    '<td><span class="wcdm-type-badge ' + log.action_type_raw + '">' + log.action_type + '</span></td>' +
                    '<td>' + (log.discount_info || '-') + '</td>' +
                    '<td><strong>' + log.total_products + '</strong></td>' +
                    '<td>' + (log.categories || '-') + '</td>' +
                    '<td>' + log.user_name + '</td>' +
                    '<td>' + log.created_at + '</td>' +
                    '<td>' + (log.scheduled_end || '-') + '</td>' +
                    '<td>' + (log.can_undo ? '<button class="wcdm-undo-btn" data-log-id="' + log.id + '">برگشت</button>' : '-') + '</td>' +
                    '</tr>'
                );
            });

            $('#wcdm-stat-total').text(data.total);
            $('#wcdm-stat-products').text(totalProducts);
            $('#wcdm-stat-undos').text(undoCount);

            this.renderPagination($pagination, data.pages, data.current_page);
        },

        renderPagination: function ($container, totalPages, currentPage) {
            $container.empty();
            if (totalPages <= 1) return;

            $container.append('<button data-page="' + (currentPage - 1) + '" ' + (currentPage === 1 ? 'disabled' : '') + '>' +
                '<span class="dashicons dashicons-arrow-right-alt2"></span></button>');

            for (let i = 1; i <= totalPages; i++) {
                if (i === 1 || i === totalPages || (i >= currentPage - 2 && i <= currentPage + 2)) {
                    $container.append('<button data-page="' + i + '" class="' + (i === currentPage ? 'active' : '') + '">' + i + '</button>');
                } else if (i === currentPage - 3 || i === currentPage + 3) {
                    $container.append('<span>...</span>');
                }
            }

            $container.append('<button data-page="' + (currentPage + 1) + '" ' + (currentPage === totalPages ? 'disabled' : '') + '>' +
                '<span class="dashicons dashicons-arrow-left-alt2"></span></button>');
        },

        handlePagination: function (e) {
            const page = $(e.currentTarget).data('page');
            if (page) this.loadLogs(page);
        },

        handleUndo: function (e) {
            const $btn = $(e.currentTarget);
            const logId = $btn.data('log-id');

            if (!confirm('آیا از برگشت این عملیات اطمینان دارید؟')) return;

            $btn.prop('disabled', true).text('...');

            this.showBatchProgress(0, 0, 0, 'شروع برگشت...');
            this.processUndoBatch(logId, 0);
        },

        processUndoBatch: function (logId, offset) {
            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_undo_batch',
                    nonce: wcdm_vars.nonce,
                    log_id: logId,
                    offset: offset
                },
                success: function (response) {
                    if (response.success) {
                        if (response.data.complete) {
                            WCDM.hideProgress();
                            WCDM.showToast(response.data.message, 'success');
                            WCDM.loadLogs(WCDM.currentPage || 1);
                            WCDM.loadRecentLogs();
                        } else {
                            WCDM.showBatchProgress(
                                response.data.progress,
                                0,
                                0,
                                'بازگردانی: ' + response.data.offset + ' از ' + response.data.total
                            );

                            setTimeout(function () {
                                WCDM.processUndoBatch(logId, response.data.offset);
                            }, 100);
                        }
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                        $('.wcdm-undo-btn[data-log-id="' + logId + '"]').prop('disabled', false).text('برگشت');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                    $('.wcdm-undo-btn[data-log-id="' + logId + '"]').prop('disabled', false).text('برگشت');
                }
            });
        },

        handleClearLogs: function () {
            if (!confirm(wcdm_vars.strings.confirm_clear_logs)) return;

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_clear_logs',
                    nonce: wcdm_vars.nonce
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.showToast(response.data.message, 'success');
                        WCDM.loadLogs(1);
                    }
                }
            });
        },

        handleExportLogs: function () {
            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_export_logs',
                    nonce: wcdm_vars.nonce
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.downloadCSV(response.data.data, 'wcdm-logs-export.csv');
                    }
                }
            });
        },

        downloadCSV: function (data, filename) {
            let csv = '\uFEFF';
            data.forEach(function (row) {
                csv += row.map(function (cell) {
                    return '"' + String(cell).replace(/"/g, '""') + '"';
                }).join(',') + '\n';
            });

            const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            link.href = URL.createObjectURL(blob);
            link.download = filename;
            link.click();
        },

        // Utility
        showProgress: function (message) {
            message = message || wcdm_vars.strings.processing;
            $('#wcdm-progress-modal .wcdm-progress-text').text(message);
            $('#wcdm-progress-modal .wcdm-progress-bar-inner').css('width', '0%').css('animation', 'wcdm-progress 1.5s ease-in-out infinite');
            $('#wcdm-progress-modal .wcdm-batch-info').hide();
            $('#wcdm-progress-modal').fadeIn(150);
        },

        showBatchProgress: function (progress, currentBatch, totalBatches, message) {
            $('#wcdm-progress-modal .wcdm-progress-text').text(message);
            $('#wcdm-progress-modal .wcdm-progress-bar-inner').css('width', progress + '%').css('animation', 'none');

            if (totalBatches > 0) {
                $('#wcdm-progress-modal .wcdm-batch-info').text('دسته ' + currentBatch + ' از ' + totalBatches).show();
            } else {
                $('#wcdm-progress-modal .wcdm-batch-info').hide();
            }

            $('#wcdm-progress-modal').fadeIn(150);
        },

        hideProgress: function () {
            $('#wcdm-progress-modal').fadeOut(150);
        },

        showResult: function (title, message) {
            $('#wcdm-result-title').text(title);
            $('#wcdm-result-message').text(message);
            $('#wcdm-result-modal').fadeIn(150);
        },

        closeModal: function () {
            $('.wcdm-modal').fadeOut(150);
        },

        // Price Increase Page Handlers
        handleIncreaseTypeChange: function () {
            const type = $('input[name="increase_action_type"]:checked').val();

            $('#wcdm-increase-category-card').hide();
            $('#wcdm-increase-product-card').hide();
            $('#wcdm-increase-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-increase-category-card').slideDown(300);
            } else if (type === 'product') {
                $('#wcdm-increase-product-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-increase-brand-card').slideDown(300);
            }

            $('#wcdm-increase-preview-card').hide();
            $('#wcdm-increase-apply-btn').prop('disabled', true);
        },

        handleIncreaseValueTypeChange: function () {
            const type = $('input[name="increase_type"]:checked').val();

            $('.wcdm-discount-type-option').removeClass('active');
            $('input[name="increase_type"]:checked').closest('.wcdm-discount-type-option').addClass('active');

            if (type === 'percentage') {
                $('#wcdm-increase-suffix').text('%');
            } else {
                $('#wcdm-increase-suffix').text('تومان');
            }

            this.validateIncreaseValue();
        },

        validateIncreaseValue: function () {
            const type = $('input[name="increase_type"]:checked').val();
            let value = parseFloat($('#wcdm-increase-value').val()) || 0;

            if (type === 'percentage' && value > 100) {
                $('#wcdm-increase-value').val(100);
            }
        },

        handleIncreasePreview: function () {
            const type = $('input[name="increase_action_type"]:checked').val();
            const categories = $('#wcdm-increase-categories').val() || [];
            const productIds = $('#wcdm-increase-products').val() || [];
            const increaseType = $('input[name="increase_type"]:checked').val();
            const increaseValue = parseFloat($('#wcdm-increase-value').val()) || 0;
            const roundingEnabled = $('#wcdm-increase-rounding-enable').is(':checked');
            const roundingType = $('input[name="increase_rounding_type"]:checked').val();
            const roundingPrecision = parseInt($('#wcdm-increase-rounding-precision').val()) || 1;

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            if (type === 'product' && productIds.length === 0) {
                this.showToast('لطفاً حداقل یک محصول انتخاب کنید.', 'warning');
                return;
            }

            if (increaseValue <= 0) {
                this.showToast('لطفاً مقدار افزایش را وارد کنید.', 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-increase-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    mode: 'increase',
                    discount_type: increaseType,
                    discount_value: increaseValue,
                    rounding_enabled: roundingEnabled,
                    rounding_type: roundingType,
                    rounding_precision: roundingPrecision
                },
                success: function (response) {
                    console.log('=== INCREASE PREVIEW REQUEST ===');
                    console.log('Sent data:', {
                        mode: 'increase',
                        discount_type: increaseType,
                        discount_value: increaseValue,
                        rounding_enabled: roundingEnabled,
                        rounding_type: roundingType,
                        rounding_precision: roundingPrecision
                    });
                    console.log('Response:', response);

                    WCDM.hideProgress();
                    if (response.success) {
                        if (response.data.products && response.data.products.length > 0) {
                            console.log('First product:', response.data.products[0]);
                        }
                        WCDM.renderIncreasePreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderIncreasePreview: function (data) {
            const $card = $('#wcdm-increase-preview-card');
            const $tbody = $('#wcdm-increase-preview-table tbody');
            const $count = $('#wcdm-increase-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="4" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-increase-apply-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    const newPrice = product.new_regular_price || product.regular_price;
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + product.regular_price + '</td>' +
                        '<td><span class="wcdm-new-price">' + newPrice + '</span></td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="4" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-increase-apply-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleIncreaseApply: function () {
            if (!confirm('آیا از افزایش قیمت محصولات اطمینان دارید؟')) return;

            const type = $('input[name="increase_action_type"]:checked').val();
            const categories = $('#wcdm-increase-categories').val() || [];
            const productIds = $('#wcdm-increase-products').val() || [];
            const increaseType = $('input[name="increase_type"]:checked').val();
            const increaseValue = parseFloat($('#wcdm-increase-value').val()) || 0;
            const roundingEnabled = $('#wcdm-increase-rounding-enable').is(':checked');
            const roundingType = $('input[name="increase_rounding_type"]:checked').val();
            const roundingPrecision = parseInt($('#wcdm-increase-rounding-precision').val()) || 1;
            const notes = $('#wcdm-increase-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-increase-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    action_type: 'increase_price',
                    discount_type: increaseType,
                    discount_value: increaseValue,
                    rounding_enabled: roundingEnabled,
                    rounding_type: roundingType,
                    rounding_precision: roundingPrecision,
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'increase');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        // Decrease Page Handlers
        handleDecreaseTypeChange: function () {
            const type = $('input[name="decrease_action_type"]:checked').val();

            $('#wcdm-decrease-category-card').hide();
            $('#wcdm-decrease-product-card').hide();
            $('#wcdm-decrease-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-decrease-category-card').slideDown(300);
            } else if (type === 'product') {
                $('#wcdm-decrease-product-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-decrease-brand-card').slideDown(300);
            }

            $('#wcdm-decrease-preview-card').hide();
            $('#wcdm-decrease-apply-btn').prop('disabled', true);
        },

        handleDecreaseValueTypeChange: function () {
            const type = $('input[name="decrease_type"]:checked').val();

            $('.wcdm-discount-type-option').removeClass('active');
            $('input[name="decrease_type"]:checked').closest('.wcdm-discount-type-option').addClass('active');

            if (type === 'percentage') {
                $('#wcdm-decrease-suffix').text('%');
            } else {
                $('#wcdm-decrease-suffix').text('تومان');
            }

            this.validateDecreaseValue();
        },

        validateDecreaseValue: function () {
            const type = $('input[name="decrease_type"]:checked').val();
            let value = parseFloat($('#wcdm-decrease-value').val()) || 0;

            if (type === 'percentage' && value > 100) {
                $('#wcdm-decrease-value').val(100);
            }
        },

        handleDecreasePreview: function () {
            const type = $('input[name="decrease_action_type"]:checked').val();
            const categories = $('#wcdm-decrease-categories').val() || [];
            const productIds = $('#wcdm-decrease-products').val() || [];
            const decreaseType = $('input[name="decrease_type"]:checked').val();
            const decreaseValue = parseFloat($('#wcdm-decrease-value').val()) || 0;
            const roundingEnabled = $('#wcdm-decrease-rounding-enable').is(':checked');
            const roundingType = $('input[name="decrease_rounding_type"]:checked').val();
            const roundingPrecision = parseInt($('#wcdm-decrease-rounding-precision').val()) || 1;

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            if (type === 'product' && productIds.length === 0) {
                this.showToast('لطفاً حداقل یک محصول انتخاب کنید.', 'warning');
                return;
            }

            if (decreaseValue <= 0) {
                this.showToast('لطفاً مقدار کاهش را وارد کنید.', 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-decrease-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    mode: 'decrease',
                    discount_type: decreaseType,
                    discount_value: decreaseValue,
                    rounding_enabled: roundingEnabled,
                    rounding_type: roundingType,
                    rounding_precision: roundingPrecision
                },
                success: function (response) {
                    WCDM.hideProgress();
                    if (response.success) {
                        WCDM.renderDecreasePreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderDecreasePreview: function (data) {
            const $card = $('#wcdm-decrease-preview-card');
            const $tbody = $('#wcdm-decrease-preview-table tbody');
            const $count = $('#wcdm-decrease-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="4" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-decrease-apply-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    const newPrice = product.new_regular_price || product.regular_price;
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + product.regular_price + '</td>' +
                        '<td><span class="wcdm-new-price">' + newPrice + '</span></td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="4" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-decrease-apply-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleDecreaseApply: function () {
            if (!confirm('آیا از کاهش قیمت محصولات اطمینان دارید؟')) return;

            const type = $('input[name="decrease_action_type"]:checked').val();
            const categories = $('#wcdm-decrease-categories').val() || [];
            const productIds = $('#wcdm-decrease-products').val() || [];
            const decreaseType = $('input[name="decrease_type"]:checked').val();
            const decreaseValue = parseFloat($('#wcdm-decrease-value').val()) || 0;
            const roundingEnabled = $('#wcdm-decrease-rounding-enable').is(':checked');
            const roundingType = $('input[name="decrease_rounding_type"]:checked').val();
            const roundingPrecision = parseInt($('#wcdm-decrease-rounding-precision').val()) || 1;
            const notes = $('#wcdm-decrease-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-decrease-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    action_type: 'decrease_price',
                    discount_type: decreaseType,
                    discount_value: decreaseValue,
                    rounding_enabled: roundingEnabled,
                    rounding_type: roundingType,
                    rounding_precision: roundingPrecision,
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'decrease');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        // Rounding Page Handlers
        handleRoundingTypeChange: function () {
            const type = $('input[name="rounding_action_type"]:checked').val();

            $('#wcdm-rounding-category-card').hide();
            $('#wcdm-rounding-product-card').hide();
            $('#wcdm-rounding-brand-card').hide();

            if (type === 'category') {
                $('#wcdm-rounding-category-card').slideDown(300);
            } else if (type === 'product') {
                $('#wcdm-rounding-product-card').slideDown(300);
            } else if (type === 'brand') {
                $('#wcdm-rounding-brand-card').slideDown(300);
            }

            $('#wcdm-rounding-preview-card').hide();
            $('#wcdm-rounding-apply-btn').prop('disabled', true);
        },

        handleRoundingDirectionChange: function () {
            $('.wcdm-discount-type-option').removeClass('active');
            $('input[name="rounding_direction"]:checked').closest('.wcdm-discount-type-option').addClass('active');
        },

        handleRoundingPriceTypeChange: function () {
            $('.wcdm-discount-type-option').removeClass('active');
            $('input[name="rounding_price_type"]:checked').closest('.wcdm-discount-type-option').addClass('active');
        },

        handleRoundingPreview: function () {
            const type = $('input[name="rounding_action_type"]:checked').val();
            const categories = $('#wcdm-rounding-categories').val() || [];
            const productIds = $('#wcdm-rounding-products').val() || [];
            const roundingType = $('input[name="rounding_direction"]:checked').val();
            const priceType = $('input[name="rounding_price_type"]:checked').val();
            const precision = parseInt($('#wcdm-rounding-precision').val()) || 1;

            if (type === 'category' && categories.length === 0) {
                this.showToast('لطفاً حداقل یک دسته‌بندی انتخاب کنید.', 'warning');
                return;
            }

            if (type === 'product' && productIds.length === 0) {
                this.showToast('لطفاً حداقل یک محصول انتخاب کنید.', 'warning');
                return;
            }

            this.showProgress('در حال بارگذاری پیش‌نمایش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_preview_products',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-rounding-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    mode: 'rounding',
                    rounding_type: roundingType,
                    price_type: priceType,
                    rounding_precision: precision
                },
                success: function (response) {
                    WCDM.hideProgress();
                    if (response.success) {
                        WCDM.renderRoundingPreview(response.data);
                    } else {
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        renderRoundingPreview: function (data) {
            const $card = $('#wcdm-rounding-preview-card');
            const $tbody = $('#wcdm-rounding-preview-table tbody');
            const $count = $('#wcdm-rounding-preview-count');

            $tbody.empty();
            $count.text(data.total);

            if (data.products.length === 0) {
                $tbody.html('<tr><td colspan="6" class="wcdm-empty-state">' +
                    '<span class="dashicons dashicons-info"></span>' +
                    '<h3>' + wcdm_vars.strings.no_products + '</h3></td></tr>');
                $('#wcdm-rounding-apply-btn').prop('disabled', true);
            } else {
                const displayProducts = data.products.slice(0, 100);
                displayProducts.forEach(function (product, index) {
                    $tbody.append(
                        '<tr class="wcdm-slide-in" style="animation-delay: ' + (index * 0.02) + 's">' +
                        '<td><a href="' + product.edit_link + '" target="_blank" class="wcdm-product-name">' + product.name + '</a></td>' +
                        '<td>' + product.regular_price + '</td>' +
                        '<td><span class="wcdm-new-price">' + (product.new_regular_price || product.regular_price) + '</span></td>' +
                        '<td>' + product.sale_price + '</td>' +
                        '<td><span class="wcdm-new-price">' + (product.new_sale_price || product.sale_price) + '</span></td>' +
                        '<td><span class="wcdm-categories-list">' + product.categories.join('، ') + '</span></td>' +
                        '</tr>'
                    );
                });

                if (data.total > 100) {
                    $tbody.append('<tr><td colspan="6" class="wcdm-more-info">... و ' + (data.total - 100) + ' محصول دیگر</td></tr>');
                }

                $('#wcdm-rounding-apply-btn').prop('disabled', false);
            }

            $card.slideDown(300);
            $('html, body').animate({ scrollTop: $card.offset().top - 50 }, 400);
        },

        handleRoundingApply: function () {
            if (!confirm('آیا از گرد کردن قیمت محصولات اطمینان دارید؟')) return;

            const type = $('input[name="rounding_action_type"]:checked').val();
            const categories = $('#wcdm-rounding-categories').val() || [];
            const productIds = $('#wcdm-rounding-products').val() || [];
            const roundingType = $('input[name="rounding_direction"]:checked').val();
            const priceType = $('input[name="rounding_price_type"]:checked').val();
            const precision = parseInt($('#wcdm-rounding-precision').val()) || 1;
            const notes = $('#wcdm-rounding-notes').val();

            this.showBatchProgress(0, 0, 0, 'شروع پردازش...');

            $.ajax({
                url: wcdm_vars.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcdm_start_batch',
                    nonce: wcdm_vars.nonce,
                    categories: categories,
                    brands: $('#wcdm-rounding-brands').val() || [],
                    product_ids: productIds,
                    all_products: type === 'all',
                    action_type: 'round_prices',
                    rounding_type: roundingType,
                    price_type: priceType,
                    rounding_precision: precision,
                    notes: notes
                },
                success: function (response) {
                    if (response.success) {
                        WCDM.processBatch(response.data.log_id, 0, response.data.total_batches, response.data.total_products, 'rounding');
                    } else {
                        WCDM.hideProgress();
                        WCDM.showToast(response.data || wcdm_vars.strings.error, 'error');
                    }
                },
                error: function () {
                    WCDM.hideProgress();
                    WCDM.showToast(wcdm_vars.strings.error, 'error');
                }
            });
        },

        showToast: function (message, type) {
            type = type || 'info';
            const iconMap = { success: 'yes-alt', error: 'warning', warning: 'info', info: 'info' };

            const $toast = $('<div class="wcdm-toast ' + type + '">' +
                '<span class="dashicons dashicons-' + iconMap[type] + '"></span>' + message + '</div>');

            $('body').append($toast);

            setTimeout(function () {
                $toast.fadeOut(300, function () { $(this).remove(); });
            }, 3500);
        }
    };

    $(document).ready(function () {
        WCDM.init();
    });

})(jQuery);
