<?php
/**
 * Plugin Name: نمایش حرفه‌ای محصولات ووکامرس
 * Plugin URI: https://mostafanoor.ir/
 * Description: افزونه نمایش محصولات ووکامرس با سفارشی‌سازی کامل، حالت‌های مختلف نمایش (Grid, Slider, Cards) و انیمیشن‌های زیبا
 * Version: 1.0.0
 * Author: Mostafa Noor
 * Author URI: https://mostafanoor.ir/
 * Text Domain: product-display-woo
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * WC requires at least: 3.0
 * WC tested up to: 8.0
 * License: GPL v2 or later
 */

// جلوگیری از دسترسی مستقیم
if (!defined('ABSPATH')) {
    exit;
}

// تعریف ثابت‌های پلاگین
define('PDW_VERSION', '1.0.0');
define('PDW_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PDW_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PDW_PLUGIN_BASENAME', plugin_basename(__FILE__));

// ========================================
// ثبت تنظیمات
// ========================================
add_action('admin_init', 'pdw_register_settings');
function pdw_register_settings() {
    register_setting('pdw_settings_group', 'pdw_settings');
}

// ========================================
// دریافت تنظیمات با مقادیر پیش‌فرض
// ========================================
function pdw_get_settings() {
    $defaults = array(
        'bg_color' => '#ffffff',
        'border_color' => '#e0e0e0',
        'border_radius' => '8',
        'card_bg' => '#ffffff',
        'card_hover_shadow' => 'rgba(0,0,0,0.08)',
        'card_border_radius' => '5',
        'button_bg' => '#e65261',
        'button_hover_bg' => '#d14450',
        'button_text_color' => '#ffffff',
        'button_border_radius' => '8',
        'button_font_size' => '16',
        'title_color' => '#333333',
        'title_font_size' => '18',
        'product_title_color' => '#444444',
        'product_title_font_size' => '14',
        'price_color' => '#2a2a2a',
        'price_font_size' => '20',
        'discount_bg' => '#ff5722',
        'discount_text_color' => '#ffffff',
        'discount_border_radius' => '20',
        'featured_bg' => '#ffc107',
        'featured_text_color' => '#333333',
    );
    
    $settings = get_option('pdw_settings', $defaults);
    return wp_parse_args($settings, $defaults);
}

// ========================================
// بارگذاری اسکریپت‌های اسلایدر
// ========================================
add_action('wp_enqueue_scripts', 'pdw_enqueue_scripts');
function pdw_enqueue_scripts() {
    wp_enqueue_style('pdw-slick', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.8.1/slick.min.css', array(), '1.8.1');
    wp_enqueue_style('pdw-slick-theme', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.8.1/slick-theme.min.css', array(), '1.8.1');
    wp_enqueue_script('pdw-slick-js', 'https://cdnjs.cloudflare.com/ajax/libs/slick-carousel/1.8.1/slick.min.js', array('jquery'), '1.8.1', true);
}

// ========================================
// شورت‌کد نمایش محصولات
// ========================================
add_shortcode('product_display', 'pdw_product_display_shortcode');
function pdw_product_display_shortcode($atts) {
    static $styles_loaded = false;
    static $instance_count = 0;
    
    $atts = shortcode_atts(array(
        'product_ids'   => '',
        'category_link' => '#',
        'top_text'      => 'خرید گوشی شیائومی از دیجی‌کالا',
        'button_text'   => 'مشاهده تمامی مدل‌ها',
        'layout'        => 'grid',
        'animation'     => 'fade',
        'show_discount' => 'yes',
        'show_stock'    => 'yes',
    ), $atts, 'product_display');

    if (!class_exists('WooCommerce')) {
        return '<div class="pdw-error"><p>برای استفاده از این بخش، افزونه ووکامرس باید نصب و فعال باشد.</p></div>';
    }

    if (empty($atts['product_ids'])) {
        return '<div class="pdw-error"><p>لطفاً شناسه محصولات را در شورت‌کد وارد کنید.</p></div>';
    }

    $product_ids = array_filter(array_map('intval', explode(',', $atts['product_ids'])));
    
    if (empty($product_ids)) {
        return '<div class="pdw-error"><p>شناسه محصولات معتبر نیست.</p></div>';
    }

    if (!$styles_loaded) {
        pdw_load_styles();
        $styles_loaded = true;
    }

    $instance_count++;
    $unique_id = 'pdw-' . $instance_count;

    $layout_class = 'pdw-layout-' . sanitize_html_class($atts['layout']);
    $animation_class = 'pdw-anim-' . sanitize_html_class($atts['animation']);

    ob_start();
    ?>
    <div class="custom-product-section-wrapper <?php echo esc_attr("$layout_class $animation_class"); ?>" id="<?php echo esc_attr($unique_id); ?>" data-layout="<?php echo esc_attr($atts['layout']); ?>">
        <div class="custom-product-card cta-card">
            <h3 class="cta-title"><?php echo esc_html($atts['top_text']); ?></h3>
            <a href="<?php echo esc_url($atts['category_link']); ?>" class="cta-button">
                <?php echo esc_html($atts['button_text']); ?>
            </a>
        </div>

        <div class="products-row <?php echo $atts['layout'] === 'slider' ? 'pdw-slider' : ''; ?>">
            <?php 
            foreach ($product_ids as $product_id) :
                $product = wc_get_product($product_id);
                if (!$product || !$product->is_visible()) {
                    continue;
                }
                
                $discount_percent = 0;
                if ($product->is_on_sale()) {
                    $regular_price = (float) $product->get_regular_price();
                    $sale_price = (float) $product->get_sale_price();
                    if ($regular_price > 0) {
                        $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
                    }
                }
                
                $stock_status = $product->get_stock_status();
                $stock_quantity = $product->get_stock_quantity();
                $low_stock_threshold = 5;
            ?>
                <div class="custom-product-card">
                    <a href="<?php echo esc_url($product->get_permalink()); ?>" class="product-link">
                        <div class="product-image-wrapper">
                            <?php echo $product->get_image('woocommerce_thumbnail'); ?>
                            
                            <?php if ($atts['show_discount'] === 'yes' && $discount_percent > 0) : ?>
                                <span class="pdw-discount-badge"><?php echo $discount_percent; ?>%</span>
                            <?php endif; ?>
                            
                            <?php if ($product->is_featured()) : ?>
                                <span class="pdw-featured-badge">ویژه</span>
                            <?php endif; ?>
                        </div>
                        
                        <h4 class="product-title"><?php echo esc_html($product->get_name()); ?></h4>
                        
                        <?php if ($atts['show_stock'] === 'yes' && $stock_quantity !== null && $stock_quantity <= $low_stock_threshold && $stock_quantity > 0) : ?>
                            <div class="pdw-stock-warning">
                                ⚠️ فقط <?php echo $stock_quantity; ?> عدد باقی مانده!
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($stock_status === 'outofstock') : ?>
                            <div class="pdw-out-of-stock">ناموجود</div>
                        <?php endif; ?>
                        
                        <div class="price">
                            <?php echo $product->get_price_html(); ?>
                        </div>
                    </a>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <?php if ($atts['layout'] === 'slider') : ?>
    <script>
    jQuery(document).ready(function($) {
        $('#<?php echo $unique_id; ?> .pdw-slider').slick({
            rtl: true,
            dots: true,
            arrows: true,
            infinite: true,
            speed: 500,
            slidesToShow: 3,
            slidesToScroll: 1,
            autoplay: true,
            autoplaySpeed: 3000,
            responsive: [
                {
                    breakpoint: 768,
                    settings: {
                        slidesToShow: 2,
                        slidesToScroll: 1
                    }
                },
                {
                    breakpoint: 480,
                    settings: {
                        slidesToShow: 1,
                        slidesToScroll: 1
                    }
                }
            ]
        });
    });
    </script>
    <?php endif; ?>
    <?php
    return ob_get_clean();
}

// ========================================
// بارگذاری استایل‌ها
// ========================================
function pdw_load_styles() {
    $s = pdw_get_settings();
    ?>
    <style type="text/css">
        .pdw-error {
            text-align: center;
            color: #dc3232;
            background: #fff8e5;
            border: 1px solid #f0c36d;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
        }
        
        .custom-product-section-wrapper {
            display: flex;
            justify-content: center;
            align-items: stretch;
            gap: 10px;
            direction: rtl;
            flex-wrap: wrap;
            border: 1px solid <?php echo esc_attr($s['border_color']); ?>;
            padding: 10px;
            border-radius: <?php echo esc_attr($s['border_radius']); ?>px;
            max-width: 100%;
            margin: 20px 0;
            box-sizing: border-box;
            position: relative;
            overflow: hidden;
            background: <?php echo esc_attr($s['bg_color']); ?>;
        }
        
        .pdw-layout-grid .products-row {
            display: contents;
        }
        
        .pdw-layout-cards .custom-product-card {
            min-width: 300px;
            padding: 20px;
        }
        .pdw-layout-cards .product-link img {
            max-height: 250px !important;
        }
        
        .pdw-layout-slider .products-row {
            display: block !important;
            width: 100%;
        }
        .pdw-layout-slider .custom-product-card {
            min-width: auto !important;
            margin: 0 5px;
        }
        .slick-prev:before, .slick-next:before {
            color: #333 !important;
        }
        .slick-dots li button:before {
            font-size: 12px !important;
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(-50px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        .pdw-anim-fade .custom-product-card {
            animation: fadeInUp 0.6s ease-out;
            animation-fill-mode: both;
        }
        
        .pdw-anim-slide .custom-product-card {
            animation: slideInRight 0.6s ease-out;
            animation-fill-mode: both;
        }
        
        .pdw-anim-fade .custom-product-card:nth-child(1) { animation-delay: 0.1s; }
        .pdw-anim-fade .custom-product-card:nth-child(2) { animation-delay: 0.2s; }
        .pdw-anim-fade .custom-product-card:nth-child(3) { animation-delay: 0.3s; }
        .pdw-anim-fade .custom-product-card:nth-child(4) { animation-delay: 0.4s; }
        .pdw-anim-fade .custom-product-card:nth-child(5) { animation-delay: 0.5s; }
        
        .pdw-anim-slide .custom-product-card:nth-child(1) { animation-delay: 0.1s; }
        .pdw-anim-slide .custom-product-card:nth-child(2) { animation-delay: 0.2s; }
        .pdw-anim-slide .custom-product-card:nth-child(3) { animation-delay: 0.3s; }
        .pdw-anim-slide .custom-product-card:nth-child(4) { animation-delay: 0.4s; }
        .pdw-anim-slide .custom-product-card:nth-child(5) { animation-delay: 0.5s; }
        
        .custom-product-card {
            flex: 1;
            min-width: 220px;
            display: flex;
            flex-direction: column;
            text-align: center;
            box-sizing: border-box;
            padding: 10px;
            border-radius: <?php echo esc_attr($s['card_border_radius']); ?>px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            background: <?php echo esc_attr($s['card_bg']); ?>;
        }
        
        .custom-product-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 4px 15px <?php echo esc_attr($s['card_hover_shadow']); ?>;
        }
        
        .custom-product-card.cta-card {
            justify-content: center;
        }
        
        .cta-card .cta-title {
            font-size: <?php echo esc_attr($s['title_font_size']); ?>px;
            font-weight: bold;
            color: <?php echo esc_attr($s['title_color']); ?>;
            margin: 0 0 15px 0;
        }
        
        .cta-card .cta-button {
            display: inline-block;
            color: <?php echo esc_attr($s['button_text_color']); ?>;
            text-decoration: none;
            border-radius: <?php echo esc_attr($s['button_border_radius']); ?>px;
            padding: 12px 25px;
            font-size: <?php echo esc_attr($s['button_font_size']); ?>px;
            font-weight: bold;
            max-width: 250px;
            margin: 0 auto;
            transition: all 0.3s ease;
            background-color: <?php echo esc_attr($s['button_bg']); ?>;
        }
        
        .cta-card .cta-button:hover {
            background-color: <?php echo esc_attr($s['button_hover_bg']); ?>;
        }
        
        .product-link {
            text-decoration: none;
            color: inherit;
            display: flex;
            flex-direction: column;
            height: 100%;
            align-items: center;
        }
        
        .product-image-wrapper {
            position: relative;
            width: 100%;
        }
        
        .product-link img {
            max-width: 100%;
            max-height: 180px;
            height: auto;
            object-fit: contain;
            margin-bottom: 10px;
        }
        
        .product-link .product-title {
            font-size: <?php echo esc_attr($s['product_title_font_size']); ?>px;
            color: <?php echo esc_attr($s['product_title_color']); ?>;
            line-height: 1.5;
            margin: 10px 0;
            text-align: center;
            width: 100%;
        }
        
        .product-link .price {
            margin-top: auto;
            font-size: <?php echo esc_attr($s['price_font_size']); ?>px;
            font-weight: 700;
            color: <?php echo esc_attr($s['price_color']); ?>;
            width: 100%;
            text-align: center;
        }
        
        .product-link .price del {
            color: #aaa;
            font-size: 0.8rem;
            margin-left: 8px;
        }
        
        .product-link .price ins {
            text-decoration: none;
        }
        
        .pdw-discount-badge {
            position: absolute;
            top: 10px;
            right: 10px;
            background: <?php echo esc_attr($s['discount_bg']); ?>;
            color: <?php echo esc_attr($s['discount_text_color']); ?>;
            padding: 5px 10px;
            border-radius: <?php echo esc_attr($s['discount_border_radius']); ?>px;
            font-size: 0.85rem;
            font-weight: bold;
            z-index: 10;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
        }
        
        .pdw-featured-badge {
            position: absolute;
            top: 10px;
            left: 10px;
            background: <?php echo esc_attr($s['featured_bg']); ?>;
            color: <?php echo esc_attr($s['featured_text_color']); ?>;
            padding: 5px 10px;
            border-radius: <?php echo esc_attr($s['discount_border_radius']); ?>px;
            font-size: 0.75rem;
            font-weight: bold;
            z-index: 10;
        }
        
        .pdw-stock-warning {
            background: #fff3cd;
            color: #856404;
            padding: 5px 10px;
            border-radius: 5px;
            font-size: 0.8rem;
            margin-bottom: 15px;
            border: 1px solid #ffeaa7;
        }
        
        .pdw-out-of-stock {
            background: #f44336;
            color: white;
            padding: 5px 10px;
            border-radius: 5px;
            font-size: 0.8rem;
            margin: 5px 0;
            font-weight: bold;
        }
        
        .woocommerce-Price-currencySymbol {
            font-size: 0.7em !important;
            font-weight: 400 !important;
            margin-right: 3px;
        }
        
        .products-row {
            display: contents;
        }

        @media (max-width: 768px) {
            .custom-product-section-wrapper {
                flex-direction: column;
                align-items: stretch;
            }
            .custom-product-card.cta-card {
                margin-bottom: 15px;
                width: 100%;
            }
            .pdw-layout-grid .products-row,
            .pdw-layout-cards .products-row {
                display: flex;
                gap: 10px;
                width: 100%;
            }
            .products-row .custom-product-card {
                flex: 1;
                min-width: 0;
            }
        }
    </style>
    <?php
}

// ========================================
// افزودن منو به پنل مدیریت
// ========================================
add_action('admin_menu', 'pdw_add_admin_menu');
function pdw_add_admin_menu() {
    add_menu_page(
        'نمایش محصولات',
        'نمایش محصولات',
        'manage_options',
        'product-display-settings',
        'pdw_admin_page',
        'dashicons-cart',
        56
    );
}

// ========================================
// اسکریپت‌های پنل ادمین
// ========================================
add_action('admin_enqueue_scripts', 'pdw_admin_scripts');
function pdw_admin_scripts($hook) {
    if ('toplevel_page_product-display-settings' !== $hook) {
        return;
    }
    
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    
    wp_add_inline_script('wp-color-picker', '
        jQuery(document).ready(function($) {
            $(".pdw-color-picker").wpColorPicker();
        });
    ');
}

// ========================================
// صفحه تنظیمات ادمین
// ========================================
function pdw_admin_page() {
    if (isset($_POST['pdw_settings_submit'])) {
        check_admin_referer('pdw_settings_save');
        
        $settings = array();
        $fields = array(
            'bg_color', 'border_color', 'border_radius', 'card_bg', 'card_hover_shadow',
            'card_border_radius', 'button_bg', 'button_hover_bg', 'button_text_color',
            'button_border_radius', 'button_font_size', 'title_color', 'title_font_size',
            'product_title_color', 'product_title_font_size', 'price_color', 'price_font_size',
            'discount_bg', 'discount_text_color', 'discount_border_radius',
            'featured_bg', 'featured_text_color'
        );
        
        foreach ($fields as $field) {
            if (isset($_POST['pdw_' . $field])) {
                $settings[$field] = sanitize_text_field($_POST['pdw_' . $field]);
            }
        }
        
        update_option('pdw_settings', $settings);
        echo '<div class="notice notice-success"><p>✅ تنظیمات با موفقیت ذخیره شد!</p></div>';
    }
    
    $s = pdw_get_settings();
    ?>
    <style>
        .pdw-admin-wrap { max-width: 1200px; margin: 20px; padding: 20px; background: #fff; border-radius: 8px; }
        .pdw-admin-header { border-bottom: 2px solid #e65261; padding-bottom: 15px; margin-bottom: 30px; }
        .pdw-section { margin-bottom: 30px; padding: 20px; background: #f9f9f9; border-radius: 5px; }
        .pdw-section h2 { color: #e65261; margin-top: 0; }
        .pdw-form-table { width: 100%; }
        .pdw-form-table th { text-align: right; padding: 15px 10px; width: 250px; font-weight: 600; }
        .pdw-form-table td { padding: 15px 10px; }
        .pdw-input { padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; width: 100px; }
        .pdw-save-btn { background: #e65261; color: white; border: none; padding: 12px 40px; font-size: 16px; border-radius: 5px; cursor: pointer; font-weight: bold; }
        .pdw-save-btn:hover { background: #d14450; }
        .submit-wrapper { text-align: center; padding: 20px; }
        .pdw-code-box { background: #2d2d2d; color: #f8f8f2; padding: 15px; border-radius: 5px; font-family: monospace; font-size: 13px; direction: ltr; text-align: left; overflow-x: auto; margin: 10px 0; }
    </style>
    
    <div class="wrap pdw-admin-wrap">
        <div class="pdw-admin-header">
            <h1>🎨 تنظیمات رنگ‌بندی و استایل محصولات</h1>
        </div>
        
        <form method="post" action="">
            <?php wp_nonce_field('pdw_settings_save'); ?>
            
            <div class="pdw-section">
                <h2>🎨 رنگ‌های پس‌زمینه</h2>
                <table class="pdw-form-table">
                    <tr>
                        <th>رنگ پس‌زمینه اصلی:</th>
                        <td><input type="text" name="pdw_bg_color" value="<?php echo esc_attr($s['bg_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>رنگ حاشیه:</th>
                        <td><input type="text" name="pdw_border_color" value="<?php echo esc_attr($s['border_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>گردی گوشه‌ها (px):</th>
                        <td><input type="number" name="pdw_border_radius" value="<?php echo esc_attr($s['border_radius']); ?>" class="pdw-input" min="0" max="50" /></td>
                    </tr>
                </table>
            </div>
            
            <div class="pdw-section">
                <h2>🃏 کارت محصول</h2>
                <table class="pdw-form-table">
                    <tr>
                        <th>رنگ پس‌زمینه کارت:</th>
                        <td><input type="text" name="pdw_card_bg" value="<?php echo esc_attr($s['card_bg']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>سایه هنگام hover:</th>
                        <td><input type="text" name="pdw_card_hover_shadow" value="<?php echo esc_attr($s['card_hover_shadow']); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th>گردی گوشه‌های کارت (px):</th>
                        <td><input type="number" name="pdw_card_border_radius" value="<?php echo esc_attr($s['card_border_radius']); ?>" class="pdw-input" min="0" max="50" /></td>
                    </tr>
                </table>
            </div>
            
            <div class="pdw-section">
                <h2>🔘 دکمه</h2>
                <table class="pdw-form-table">
                    <tr>
                        <th>رنگ دکمه:</th>
                        <td><input type="text" name="pdw_button_bg" value="<?php echo esc_attr($s['button_bg']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>رنگ دکمه (hover):</th>
                        <td><input type="text" name="pdw_button_hover_bg" value="<?php echo esc_attr($s['button_hover_bg']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>رنگ متن دکمه:</th>
                        <td><input type="text" name="pdw_button_text_color" value="<?php echo esc_attr($s['button_text_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>گردی دکمه (px):</th>
                        <td><input type="number" name="pdw_button_border_radius" value="<?php echo esc_attr($s['button_border_radius']); ?>" class="pdw-input" min="0" max="50" /></td>
                    </tr>
                    <tr>
                        <th>سایز فونت دکمه (px):</th>
                        <td><input type="number" name="pdw_button_font_size" value="<?php echo esc_attr($s['button_font_size']); ?>" class="pdw-input" min="10" max="30" /></td>
                    </tr>
                </table>
            </div>
            
            <div class="pdw-section">
                <h2>📝 متن‌ها</h2>
                <table class="pdw-form-table">
                    <tr>
                        <th>رنگ عنوان اصلی:</th>
                        <td><input type="text" name="pdw_title_color" value="<?php echo esc_attr($s['title_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>سایز عنوان اصلی (px):</th>
                        <td><input type="number" name="pdw_title_font_size" value="<?php echo esc_attr($s['title_font_size']); ?>" class="pdw-input" min="12" max="36" /></td>
                    </tr>
                    <tr>
                        <th>رنگ عنوان محصول:</th>
                        <td><input type="text" name="pdw_product_title_color" value="<?php echo esc_attr($s['product_title_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>سایز عنوان محصول (px):</th>
                        <td><input type="number" name="pdw_product_title_font_size" value="<?php echo esc_attr($s['product_title_font_size']); ?>" class="pdw-input" min="10" max="24" /></td>
                    </tr>
                    <tr>
                        <th>رنگ قیمت:</th>
                        <td><input type="text" name="pdw_price_color" value="<?php echo esc_attr($s['price_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>سایز قیمت (px):</th>
                        <td><input type="number" name="pdw_price_font_size" value="<?php echo esc_attr($s['price_font_size']); ?>" class="pdw-input" min="12" max="32" /></td>
                    </tr>
                </table>
            </div>
            
            <div class="pdw-section">
                <h2>🏷️ بج‌ها</h2>
                <table class="pdw-form-table">
                    <tr>
                        <th>رنگ بج تخفیف:</th>
                        <td><input type="text" name="pdw_discount_bg" value="<?php echo esc_attr($s['discount_bg']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>رنگ متن بج تخفیف:</th>
                        <td><input type="text" name="pdw_discount_text_color" value="<?php echo esc_attr($s['discount_text_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>گردی بج‌ها (px):</th>
                        <td><input type="number" name="pdw_discount_border_radius" value="<?php echo esc_attr($s['discount_border_radius']); ?>" class="pdw-input" min="0" max="50" /></td>
                    </tr>
                    <tr>
                        <th>رنگ بج ویژه:</th>
                        <td><input type="text" name="pdw_featured_bg" value="<?php echo esc_attr($s['featured_bg']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                    <tr>
                        <th>رنگ متن بج ویژه:</th>
                        <td><input type="text" name="pdw_featured_text_color" value="<?php echo esc_attr($s['featured_text_color']); ?>" class="pdw-color-picker" /></td>
                    </tr>
                </table>
            </div>
            
            <div class="submit-wrapper">
                <button type="submit" name="pdw_settings_submit" class="pdw-save-btn">💾 ذخیره تنظیمات</button>
            </div>
        </form>
        
        <div class="pdw-section">
            <h2>📖 نحوه استفاده</h2>
            
            <h3>حالت شبکه‌ای (Grid):</h3>
            <div class="pdw-code-box">
                <code>[product_display product_ids="101,102,103" layout="grid" animation="fade" category_link="https://mysite.com/mobiles/" top_text="جدیدترین گوشی‌ها" button_text="مشاهده همه"]</code>
            </div>
            
            <h3>حالت اسلایدری (Slider):</h3>
            <div class="pdw-code-box">
                <code>[product_display product_ids="101,102,103" layout="slider" animation="fade" category_link="https://mysite.com/mobiles/" top_text="جدیدترین گوشی‌ها" button_text="مشاهده همه"]</code>
            </div>
            
            <h3>حالت کارت‌های بزرگ (Cards):</h3>
            <div class="pdw-code-box">
                <code>[product_display product_ids="301,302" layout="cards" category_link="https://mysite.com/collection/" top_text="کلکسیون بهار" button_text="خرید از کلکسیون"]</code>
            </div>
            
            <h3>پارامترها:</h3>
            <ul style="line-height: 2;">
                <li><strong>product_ids:</strong> شناسه محصولات (الزامی) - مثال: 101,102,103</li>
                <li><strong>layout:</strong> grid یا slider یا cards</li>
                <li><strong>animation:</strong> fade یا slide یا none</li>
                <li><strong>category_link:</strong> لینک دکمه</li>
                <li><strong>top_text:</strong> متن عنوان</li>
                <li><strong>button_text:</strong> متن دکمه</li>
                <li><strong>show_discount:</strong> yes یا no</li>
                <li><strong>show_stock:</strong> yes یا no</li>
            </ul>
        </div>
    </div>
    <?php
}

// ========================================
// لینک تنظیمات در صفحه افزونه‌ها
// ========================================
add_filter('plugin_action_links_' . PDW_PLUGIN_BASENAME, 'pdw_add_settings_link');
function pdw_add_settings_link($links) {
    $settings_link = '<a href="admin.php?page=product-display-settings">تنظیمات</a>';
    array_unshift($links, $settings_link);
    return $links;
}

// ========================================
// هوک فعال‌سازی پلاگین
// ========================================
register_activation_hook(__FILE__, 'pdw_activate_plugin');
function pdw_activate_plugin() {
    // ایجاد تنظیمات پیش‌فرض در فعال‌سازی
    if (!get_option('pdw_settings')) {
        $defaults = array(
            'bg_color' => '#ffffff',
            'border_color' => '#e0e0e0',
            'border_radius' => '8',
            'card_bg' => '#ffffff',
            'card_hover_shadow' => 'rgba(0,0,0,0.08)',
            'card_border_radius' => '5',
            'button_bg' => '#e65261',
            'button_hover_bg' => '#d14450',
            'button_text_color' => '#ffffff',
            'button_border_radius' => '8',
            'button_font_size' => '16',
            'title_color' => '#333333',
            'title_font_size' => '18',
            'product_title_color' => '#444444',
            'product_title_font_size' => '14',
            'price_color' => '#2a2a2a',
            'price_font_size' => '20',
            'discount_bg' => '#ff5722',
            'discount_text_color' => '#ffffff',
            'discount_border_radius' => '20',
            'featured_bg' => '#ffc107',
            'featured_text_color' => '#333333',
        );
        add_option('pdw_settings', $defaults);
    }
}

// ========================================
// هوک غیرفعال‌سازی پلاگین
// ========================================
register_deactivation_hook(__FILE__, 'pdw_deactivate_plugin');
function pdw_deactivate_plugin() {
    // می‌توانید در اینجا کارهای لازم برای غیرفعال‌سازی را انجام دهید
}

// ========================================
// پاک کردن تنظیمات هنگام حذف پلاگین
// ========================================
register_uninstall_hook(__FILE__, 'pdw_uninstall_plugin');
function pdw_uninstall_plugin() {
    // حذف تنظیمات از دیتابیس
    delete_option('pdw_settings');
}