<?php
/**
 * Plugin Name: پلاگین تخفیف بر اساس تعداد محصولات
 * Plugin URI: https://your-website.com
 * Description: ایجاد تخفیف خودکار بر اساس تعداد محصولات در سبد خرید
 * Version: 1.0.1
 * Author: Your Name
 * Text Domain: quantity-discount
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.3
 * WC requires at least: 3.0
 * WC tested up to: 8.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define constants
define('QD_PLUGIN_URL', plugin_dir_url(__FILE__));
define('QD_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('QD_VERSION', '1.0.1');

class WC_Quantity_Discount_Plugin {
    
    private $plugin_name = 'quantity-discount';
    private $version = '1.0.1';
    
    public function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Hook into WooCommerce
        add_action('woocommerce_cart_calculate_fees', array($this, 'add_quantity_discount'), 20, 1);
        add_action('woocommerce_after_cart_table', array($this, 'display_discount_message'));
        add_action('woocommerce_widget_shopping_cart_total', array($this, 'display_mini_cart_discount'));
        
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_add_discount_rule', array($this, 'add_discount_rule'));
        add_action('wp_ajax_delete_discount_rule', array($this, 'delete_discount_rule'));
        
        // Enqueue styles and scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    public function woocommerce_missing_notice() {
        echo '<div class="notice notice-error"><p>';
        echo '<strong>پلاگین تخفیف بر اساس تعداد محصولات</strong> نیاز به فعال بودن WooCommerce دارد.';
        echo '</p></div>';
    }
    
    public function activate() {
        // Create default rules if they don't exist
        $default_rules = array(
            array(
                'quantity' => 2,
                'discount_type' => 'percentage',
                'discount_value' => 5,
                'message' => 'تخفیف 5% برای خرید 2 محصول'
            ),
            array(
                'quantity' => 4,
                'discount_type' => 'percentage',
                'discount_value' => 10,
                'message' => 'تخفیف 10% برای خرید 4 محصول'
            ),
            array(
                'quantity' => 5,
                'discount_type' => 'percentage',
                'discount_value' => 15,
                'message' => 'تخفیف 15% برای خرید 5 محصول'
            )
        );
        
        if (!get_option('quantity_discount_rules')) {
            add_option('quantity_discount_rules', $default_rules);
        }
        
        // Create CSS file
        $this->create_assets();
    }
    
    public function deactivate() {
        // Clean up if needed
    }
    
    private function create_assets() {
        $upload_dir = wp_upload_dir();
        $plugin_dir = $upload_dir['basedir'] . '/quantity-discount/';
        
        if (!file_exists($plugin_dir)) {
            wp_mkdir_p($plugin_dir);
        }
        
        $css_content = '
/* تخفیف در سبد خرید */
.quantity-discount-message-wrapper {
    margin: 20px 0;
}

.discount-notification {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 15px 20px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    gap: 10px;
    margin: 10px 0;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    animation: slideInDown 0.5s ease-out;
}

.discount-notification .discount-icon {
    font-size: 24px;
    animation: bounce 2s infinite;
}

.discount-notification .discount-text {
    flex: 1;
    font-weight: 600;
    font-size: 14px;
}

.discount-notification .discount-amount {
    background: rgba(255, 255, 255, 0.2);
    padding: 5px 12px;
    border-radius: 20px;
    font-weight: bold;
    backdrop-filter: blur(10px);
}

.next-discount-info {
    background: linear-gradient(45deg, #feca57, #ff9ff3);
    color: #333;
    padding: 12px 18px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    gap: 10px;
    margin: 10px 0;
    font-weight: 600;
    font-size: 14px;
    box-shadow: 0 3px 10px rgba(254, 202, 87, 0.3);
}

.next-discount-info .info-icon {
    font-size: 20px;
}

.mini-cart-discount {
    background: linear-gradient(45deg, #ff6b6b, #ee5a24);
    color: white;
    padding: 8px 12px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    gap: 8px;
    margin: 8px 0;
    font-size: 12px;
    font-weight: 600;
    box-shadow: 0 2px 8px rgba(255, 107, 107, 0.3);
}

.mini-cart-discount .discount-icon {
    font-size: 16px;
}

@keyframes slideInDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

@keyframes bounce {
    0%, 20%, 50%, 80%, 100% {
        transform: translateY(0);
    }
    40% {
        transform: translateY(-5px);
    }
    60% {
        transform: translateY(-3px);
    }
}

@media (max-width: 768px) {
    .discount-notification {
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .discount-notification .discount-text {
        font-size: 13px;
    }
    
    .next-discount-info {
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .mini-cart-discount {
        font-size: 11px;
        padding: 6px 10px;
    }
}
';
        
        file_put_contents($plugin_dir . 'style.css', $css_content);
    }
    
    public function add_quantity_discount($cart) {
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }
        
        if (!$cart) {
            return;
        }
        
        // Prevent infinite loops
        static $discount_added = false;
        if ($discount_added) {
            return;
        }
        $discount_added = true;
        
        // Calculate total quantity
        $cart_count = 0;
        foreach ($cart->get_cart() as $cart_item) {
            $cart_count += intval($cart_item['quantity']);
        }
        
        // Get discount rules
        $rules = get_option('quantity_discount_rules', array());
        if (empty($rules)) {
            $discount_added = false;
            return;
        }
        
        // Sort rules by quantity descending
        usort($rules, function($a, $b) {
            return intval($b['quantity']) - intval($a['quantity']);
        });
        
        // Find applicable discount
        $applied_discount = null;
        foreach ($rules as $rule) {
            if ($cart_count >= intval($rule['quantity'])) {
                $applied_discount = $rule;
                break;
            }
        }
        
        if ($applied_discount) {
            $cart_total = $cart->get_subtotal();
            
            if ($applied_discount['discount_type'] === 'percentage') {
                $discount_amount = ($cart_total * floatval($applied_discount['discount_value'])) / 100;
            } else {
                $discount_amount = floatval($applied_discount['discount_value']);
            }
            
            // Add discount fee
            $cart->add_fee('تخفیف بر اساس تعداد محصولات', -$discount_amount);
            
            // Store in session for display
            if (WC()->session) {
                WC()->session->set('applied_quantity_discount', $applied_discount);
                WC()->session->set('applied_discount_amount', $discount_amount);
                WC()->session->set('cart_item_count', $cart_count);
            }
        } else {
            // Clear session if no discount applies
            if (WC()->session) {
                WC()->session->__unset('applied_quantity_discount');
                WC()->session->__unset('applied_discount_amount');
                WC()->session->__unset('cart_item_count');
            }
        }
        
        $discount_added = false;
    }
    
    public function display_discount_message() {
        if (!WC()->session) {
            return;
        }
        
        $applied_discount = WC()->session->get('applied_quantity_discount');
        $discount_amount = WC()->session->get('applied_discount_amount');
        
        if ($applied_discount && $discount_amount) {
            $formatted_amount = wc_price($discount_amount);
            echo '<div class="quantity-discount-message-wrapper">
                    <div class="discount-notification">
                        <span class="discount-icon">🎉</span>
                        <span class="discount-text">' . esc_html($applied_discount['message']) . '</span>
                        <span class="discount-amount">مقدار تخفیف: ' . $formatted_amount . '</span>
                    </div>
                  </div>';
        }
        
        // Show next discount opportunity
        $this->show_next_discount_info();
    }
    
    public function show_next_discount_info() {
        if (!WC()->cart) {
            return;
        }
        
        $current_count = WC()->cart->get_cart_contents_count();
        $rules = get_option('quantity_discount_rules', array());
        
        if (empty($rules)) {
            return;
        }
        
        // Sort rules by quantity ascending
        usort($rules, function($a, $b) {
            return intval($a['quantity']) - intval($b['quantity']);
        });
        
        $next_discount = null;
        foreach ($rules as $rule) {
            if ($current_count < intval($rule['quantity'])) {
                $next_discount = $rule;
                break;
            }
        }
        
        if ($next_discount) {
            $needed = intval($next_discount['quantity']) - $current_count;
            echo '<div class="next-discount-info">
                    <span class="info-icon">💡</span>
                    <span>با اضافه کردن ' . $needed . ' محصول دیگر، ' . esc_html($next_discount['message']) . ' دریافت کنید!</span>
                  </div>';
        }
    }
    
    public function display_mini_cart_discount() {
        if (!WC()->session) {
            return;
        }
        
        $applied_discount = WC()->session->get('applied_quantity_discount');
        $discount_amount = WC()->session->get('applied_discount_amount');
        
        if ($applied_discount && $discount_amount) {
            $formatted_amount = wc_price($discount_amount);
            echo '<div class="mini-cart-discount">
                    <span class="discount-icon">🎁</span>
                    <span>تخفیف شما: ' . $formatted_amount . '</span>
                  </div>';
        }
    }
    
    public function enqueue_frontend_scripts() {
        if (is_cart() || is_checkout() || is_woocommerce()) {
            $upload_dir = wp_upload_dir();
            $css_url = $upload_dir['baseurl'] . '/quantity-discount/style.css';
            
            if (file_exists($upload_dir['basedir'] . '/quantity-discount/style.css')) {
                wp_enqueue_style(
                    'quantity-discount-style',
                    $css_url,
                    array(),
                    $this->version
                );
            }
        }
    }
    
    public function enqueue_admin_scripts($hook) {
        if ('woocommerce_page_quantity-discount-settings' === $hook) {
            wp_enqueue_script('jquery');
        }
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            'تنظیمات تخفیف بر اساس تعداد',
            'تخفیف تعدادی',
            'manage_woocommerce',
            'quantity-discount-settings',
            array($this, 'admin_page')
        );
    }
    
    public function register_settings() {
        register_setting('quantity_discount_settings', 'quantity_discount_rules');
    }
    
    public function admin_page() {
        $rules = get_option('quantity_discount_rules', array());
        ?>
        <div class="wrap">
            <h1>تنظیمات تخفیف بر اساس تعداد محصولات</h1>
            
            <div class="notice notice-info">
                <p><strong>نکته:</strong> تخفیف بر اساس تعداد کل محصولات در سبد خرید محاسبه می‌شود.</p>
            </div>
            
            <div class="quantity-discount-admin">
                <div class="add-rule-section">
                    <h2>افزودن قانون جدید</h2>
                    <table class="form-table">
                        <tr>
                            <th>تعداد محصولات</th>
                            <td><input type="number" id="rule-quantity" min="1" value="1"></td>
                        </tr>
                        <tr>
                            <th>نوع تخفیف</th>
                            <td>
                                <select id="rule-discount-type">
                                    <option value="percentage">درصدی</option>
                                    <option value="fixed">مقدار ثابت (تومان)</option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th>مقدار تخفیف</th>
                            <td><input type="number" id="rule-discount-value" min="0" step="0.01"></td>
                        </tr>
                        <tr>
                            <th>پیام نمایشی</th>
                            <td><input type="text" id="rule-message" class="regular-text" placeholder="پیام تخفیف شما"></td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="button" class="button-primary" id="add-rule-btn">افزودن قانون</button>
                    </p>
                </div>
                
                <div class="rules-list-section">
                    <h2>قوانین موجود</h2>
                    <?php if (empty($rules)): ?>
                        <p>هیچ قانونی تعریف نشده است.</p>
                    <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>تعداد محصولات</th>
                                <th>نوع تخفیف</th>
                                <th>مقدار تخفیف</th>
                                <th>پیام</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($rules as $index => $rule): ?>
                            <tr>
                                <td><?php echo esc_html($rule['quantity']); ?></td>
                                <td><?php echo $rule['discount_type'] === 'percentage' ? 'درصدی' : 'مقدار ثابت'; ?></td>
                                <td>
                                    <?php 
                                    if ($rule['discount_type'] === 'percentage') {
                                        echo esc_html($rule['discount_value']) . '%';
                                    } else {
                                        echo esc_html($rule['discount_value']) . ' تومان';
                                    }
                                    ?>
                                </td>
                                <td><?php echo esc_html($rule['message']); ?></td>
                                <td>
                                    <button type="button" class="button delete-rule" data-index="<?php echo $index; ?>">حذف</button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <style>
        .quantity-discount-admin {
            max-width: 800px;
        }
        .add-rule-section, .rules-list-section {
            background: #fff;
            padding: 20px;
            margin: 20px 0;
            border: 1px solid #ccd0d4;
            box-shadow: 0 1px 1px rgba(0,0,0,0.04);
        }
        .delete-rule {
            color: #a00;
        }
        .delete-rule:hover {
            color: #dc3232;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#add-rule-btn').on('click', function() {
                var quantity = $('#rule-quantity').val();
                var discountType = $('#rule-discount-type').val();
                var discountValue = $('#rule-discount-value').val();
                var message = $('#rule-message').val();
                
                if (!quantity || !discountValue || !message) {
                    alert('لطفا تمام فیلدها را پر کنید');
                    return;
                }
                
                $.post(ajaxurl, {
                    action: 'add_discount_rule',
                    nonce: '<?php echo wp_create_nonce('quantity_discount_nonce'); ?>',
                    quantity: quantity,
                    discount_type: discountType,
                    discount_value: discountValue,
                    message: message
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('خطا در افزودن قانون');
                    }
                });
            });
            
            $('.delete-rule').on('click', function() {
                var index = $(this).data('index');
                if (confirm('آیا از حذف این قانون مطمئن هستید؟')) {
                    $.post(ajaxurl, {
                        action: 'delete_discount_rule',
                        nonce: '<?php echo wp_create_nonce('quantity_discount_nonce'); ?>',
                        index: index
                    }, function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert('خطا در حذف قانون');
                        }
                    });
                }
            });
        });
        </script>
        <?php
    }
    
    public function add_discount_rule() {
        if (!check_ajax_referer('quantity_discount_nonce', 'nonce', false)) {
            wp_send_json_error('امنیت نامعتبر');
            return;
        }
        
        $quantity = intval($_POST['quantity']);
        $discount_type = sanitize_text_field($_POST['discount_type']);
        $discount_value = floatval($_POST['discount_value']);
        $message = sanitize_text_field($_POST['message']);
        
        if ($quantity <= 0 || $discount_value <= 0 || empty($message)) {
            wp_send_json_error('داده‌های نامعتبر');
            return;
        }
        
        $rules = get_option('quantity_discount_rules', array());
        
        $new_rule = array(
            'quantity' => $quantity,
            'discount_type' => $discount_type,
            'discount_value' => $discount_value,
            'message' => $message
        );
        
        $rules[] = $new_rule;
        update_option('quantity_discount_rules', $rules);
        
        wp_send_json_success();
    }
    
    public function delete_discount_rule() {
        if (!check_ajax_referer('quantity_discount_nonce', 'nonce', false)) {
            wp_send_json_error('امنیت نامعتبر');
            return;
        }
        
        $rules = get_option('quantity_discount_rules', array());
        $index = intval($_POST['index']);
        
        if (isset($rules[$index])) {
            unset($rules[$index]);
            $rules = array_values($rules);
            update_option('quantity_discount_rules', $rules);
            wp_send_json_success();
        } else {
            wp_send_json_error('قانون یافت نشد');
        }
    }
}

// Initialize plugin
new WC_Quantity_Discount_Plugin();