<?php
/**
 * Plugin Name: Product Category Manager
 * Description: A plugin to manage product categories in bulk by searching products by name or category and removing products from multiple categories.
 * Version: 1.5
 * Author: Mustafa Nourmohammadi
 */

// Prevent direct access to the file
if (!defined('ABSPATH')) {
    exit;
}

// Add admin menu
add_action('admin_menu', 'pcm_add_admin_menu');
function pcm_add_admin_menu() {
    add_menu_page('Product Category Manager', 'Product Category Manager', 'manage_options', 'product-category-manager', 'pcm_admin_page');
}

// Admin page content
function pcm_admin_page() {
    ?>
    <div class="wrap">
        <h1>Product Category Manager</h1>
        <form method="get">
            <input type="hidden" name="page" value="product-category-manager">
            <input type="text" name="search_product_name" placeholder="Search product by name">
            
            <!-- Category filter dropdown -->
            <select name="filter_by_category">
                <option value="">Filter by category</option>
                <?php
                $categories = get_terms('product_cat', ['hide_empty' => false]);
                foreach ($categories as $category) {
                    echo '<option value="' . esc_attr($category->term_id) . '">' . esc_html($category->name) . '</option>';
                }
                ?>
            </select>
            
            <button type="submit" class="button button-primary">Search</button>
        </form>
        <?php pcm_handle_search(); ?>
    </div>
    <?php
}

// Handle search and display results
function pcm_handle_search() {
    if (isset($_GET['search_product_name']) || isset($_GET['filter_by_category'])) {
        $search_product_name = sanitize_text_field($_GET['search_product_name']);
        $filter_by_category = sanitize_text_field($_GET['filter_by_category']);

        // Query products based on search terms and category filter
        $args = [
            'post_type' => 'product',
            'posts_per_page' => -1,
            's' => $search_product_name,
        ];

        if ($filter_by_category) {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $filter_by_category,
                ]
            ];
        }

        $products = new WP_Query($args);

        if ($products->have_posts()) {
            ?>
            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                <input type="hidden" name="action" value="pcm_bulk_action">
                <table class="wp-list-table widefat fixed striped table-view-list">
                    <thead>
                        <tr>
                            <th class="manage-column column-cb check-column"><input type="checkbox" id="select-all"></th>
                            <th>Product Name</th>
                            <th>Categories</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($products->have_posts()) : $products->the_post(); ?>
                            <tr>
                                <th scope="row" class="check-column"><input type="checkbox" name="product_ids[]" value="<?php echo get_the_ID(); ?>"></th>
                                <td><?php the_title(); ?></td>
                                <td><?php echo wc_get_product_category_list(get_the_ID()); ?></td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>

                <h3>Select categories to remove from selected products:</h3>
                <div style="height: 200px; overflow-y: auto; border: 1px solid #ccc; padding: 10px;">
                    <?php
                    $categories = get_terms('product_cat', ['hide_empty' => false]);
                    foreach ($categories as $category) {
                        echo '<label><input type="checkbox" name="categories_to_remove[]" value="' . esc_attr($category->term_id) . '"> ' . esc_html($category->name) . '</label><br>';
                    }
                    ?>
                </div>
                <button type="submit" class="button button-primary">Apply</button>
            </form>
            <?php
        } else {
            echo '<p>No products found for the search terms.</p>';
        }
    }
}

// Handle bulk action
add_action('admin_post_pcm_bulk_action', 'pcm_bulk_action');
function pcm_bulk_action() {
    if (isset($_POST['product_ids']) && isset($_POST['categories_to_remove'])) {
        $product_ids = array_map('intval', $_POST['product_ids']);
        $categories_to_remove = array_map('intval', $_POST['categories_to_remove']);

        foreach ($product_ids as $product_id) {
            // Get the current categories of the product
            $current_categories = wp_get_post_terms($product_id, 'product_cat', ['fields' => 'ids']);

            // Remove only the selected categories
            $new_categories = array_diff($current_categories, $categories_to_remove);

            // Update the product with the new category list
            wp_set_post_terms($product_id, $new_categories, 'product_cat');
        }

        wp_redirect(admin_url('admin.php?page=product-category-manager&success=1'));
        exit;
    }
}

// Include jQuery for select-all functionality
add_action('admin_footer', 'pcm_admin_footer_script');
function pcm_admin_footer_script() {
    ?>
    <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#select-all').click(function() {
                $('input[name="product_ids[]"]').prop('checked', this.checked);
            });
        });
    </script>
    <?php
}
