<?php
/**
 * Plugin Name: Custom Product Sorting
 * Description: Add custom sorting options for WooCommerce products with enhanced features
 * Version: 1.1.1
 * Author: مصطفی نور 
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Custom_Product_Sorting {
    private static $instance = null;
    private $option_name = 'custom_sort_options';

    // Singleton pattern
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Check if WooCommerce is active
        if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }

        $this->init_hooks();
    }

    private function init_hooks() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Product hooks
        add_action('woocommerce_product_options_general_product_data', array($this, 'add_custom_sort_fields'));
        add_action('woocommerce_process_product_meta', array($this, 'save_custom_sort_fields'));
        
        // Frontend hooks
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_shortcode('custom_sort_dropdown', array($this, 'render_sort_dropdown'));
        add_filter('woocommerce_get_catalog_ordering_args', array($this, 'customize_ordering_args'), 20);
        
        // Ajax handlers
        add_action('wp_ajax_save_custom_sort', array($this, 'save_custom_sort'));
        add_action('wp_ajax_delete_custom_sort', array($this, 'delete_custom_sort'));
        add_action('wp_ajax_clear_sort_values', array($this, 'clear_sort_values'));
        
        // Add sorting to default WooCommerce dropdown
        add_filter('woocommerce_catalog_orderby', array($this, 'add_sorting_to_default_dropdown'));

        // Add styles
        add_action('wp_head', array($this, 'add_custom_styles'));
    }

    public function woocommerce_missing_notice() {
        ?>
        <div class="error">
            <p><?php _e('Custom Product Sorting requires WooCommerce to be installed and activated.', 'custom-product-sorting'); ?></p>
        </div>
        <?php
    }

    public function add_custom_styles() {
        ?>
        <style>
            .custom-sort-dropdown {
                padding: 8px 12px;
                border: 1px solid #ddd;
                border-radius: 4px;
                background-color: #fff;
                font-size: 14px;
                margin: 10px 0;
            }
            .custom-sort-dropdown option:checked {
                background-color: #0073aa;
                color: white;
            }
            .custom-sort-dropdown option {
                padding: 8px;
            }
            .clear-values {
                margin-left: 5px !important;
            }
        </style>
        <?php
    }

    public function enqueue_admin_assets($hook) {
        if ('toplevel_page_custom-sort-settings' !== $hook) {
            return;
        }

        wp_enqueue_style('custom-sort-admin', plugins_url('css/admin.css', __FILE__));
        wp_enqueue_script('custom-sort-admin', plugins_url('js/admin.js', __FILE__), array('jquery'), '1.0.0', true);
    }

    public function enqueue_frontend_assets() {
        if (!is_shop() && !is_product_category() && !is_product_tag()) {
            return;
        }

        wp_enqueue_style('custom-sort-frontend', plugins_url('css/frontend.css', __FILE__));
    }

    public function add_admin_menu() {
        add_menu_page(
            'Custom Sort Options',
            'Custom Sort',
            'manage_options',
            'custom-sort-settings',
            array($this, 'render_admin_page'),
            'dashicons-sort',
            56
        );
    }

    public function render_admin_page() {
        ?>
        <div class="wrap">
            <h1>مدیریت گزینه‌های مرتب‌سازی</h1>
            <form id="add-sort-form">
                <table class="form-table">
                    <tr>
                        <th><label for="sort_key">کلید مرتب‌سازی:</label></th>
                        <td>
                            <input type="text" id="sort_key" name="sort_key" required>
                            <p class="description">مثال: best_graphics</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="sort_label">عنوان نمایشی:</label></th>
                        <td>
                            <input type="text" id="sort_label" name="sort_label" required>
                            <p class="description">مثال: بهترین گرافیک‌ها</p>
                        </td>
                    </tr>
                </table>
                <?php wp_nonce_field('custom-sort-nonce', 'custom_sort_nonce'); ?>
                <button type="submit" class="button button-primary">افزودن گزینه مرتب‌سازی</button>
            </form>
            
            <h2>گزینه‌های موجود</h2>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>کلید</th>
                        <th>عنوان</th>
                        <th>عملیات</th>
                    </tr>
                </thead>
                <tbody id="sort-options-list">
                    <?php
                    $saved_sorts = get_option('custom_sort_options', array());
                    foreach ($saved_sorts as $key => $label) {
                        echo "<tr>
                                <td>{$key}</td>
                                <td>{$label}</td>
                                <td>
                                    <button class='button delete-sort' data-key='{$key}' data-nonce='" . wp_create_nonce('delete_sort_' . $key) . "'>حذف</button>
                                    <button class='button clear-values' data-key='{$key}' data-nonce='" . wp_create_nonce('clear_values_' . $key) . "'>پاک کردن مقادیر</button>
                                </td>
                            </tr>";
                    }
                    ?>
                </tbody>
            </table>
        </div>
        <script>
        jQuery(document).ready(function($) {
            // Add new sort option
            $('#add-sort-form').on('submit', function(e) {
                e.preventDefault();
                var sortKey = $('#sort_key').val();
                var sortLabel = $('#sort_label').val();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'save_custom_sort',
                        sort_key: sortKey,
                        sort_label: sortLabel,
                        nonce: $('#custom_sort_nonce').val()
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        }
                    }
                });
            });

            // Delete sort option
            $('.delete-sort').on('click', function(e) {
                e.preventDefault();
                var button = $(this);
                var key = button.data('key');
                var nonce = button.data('nonce');
                
                if (confirm('آیا مطمئن هستید که می‌خواهید این گزینه را حذف کنید؟')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'delete_custom_sort',
                            sort_key: key,
                            nonce: nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                button.closest('tr').fadeOut(400, function() {
                                    $(this).remove();
                                });
                            }
                        }
                    });
                }
            });

            // Clear values
            $('.clear-values').on('click', function(e) {
                e.preventDefault();
                var button = $(this);
                var key = button.data('key');
                var nonce = button.data('nonce');
                
                if (confirm('آیا مطمئن هستید که می‌خواهید تمام مقادیر این گزینه مرتب‌سازی را پاک کنید؟')) {
                    button.text('در حال پاک کردن...').prop('disabled', true);
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'clear_sort_values',
                            sort_key: key,
                            nonce: nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                alert('مقادیر با موفقیت پاک شدند');
                            } else {
                                alert('خطا در پاک کردن مقادیر: ' + response.data);
                            }
                            button.text('پاک کردن مقادیر').prop('disabled', false);
                        },
                        error: function() {
                            alert('خطا در ارتباط با سرور');
                            button.text('پاک کردن مقادیر').prop('disabled', false);
                        }
                    });
                }
            });
        });
        </script>
        <?php
    }

    public function add_custom_sort_fields() {
        global $post;
        $saved_sorts = get_option($this->option_name, array());
        
        echo '<div class="options_group">';
        
        foreach ($saved_sorts as $key => $label) {
            $value = get_post_meta($post->ID, '_' . $key, true);
            
            woocommerce_wp_text_input(array(
                'id' => $key,
                'label' => $label,
                'desc_tip' => true,
                'description' => sprintf('Enter sort value for %s', $label),
                'type' => 'number',
                'custom_attributes' => array(
                    'step' => 'any',
                    'min' => '0'
                ),
                'value' => $value
            ));
        }
        
        echo '</div>';
    }

    public function save_custom_sort_fields($post_id) {
        $saved_sorts = get_option($this->option_name, array());
        
        foreach ($saved_sorts as $key => $label) {
            $meta_key = '_' . $key;
            if (isset($_POST[$key])) {
                $value = sanitize_text_field($_POST[$key]);
                update_post_meta($post_id, $meta_key, $value);
            }
        }
    }

    public function render_sort_dropdown($atts) {
        $saved_sorts = get_option($this->option_name, array());
        $current_sort = isset($_GET['custom_sort']) ? sanitize_text_field($_GET['custom_sort']) : '';
        
        // Cache the output
        $cache_key = 'custom_sort_dropdown_' . md5(serialize($saved_sorts) . $current_sort);
        $output = wp_cache_get($cache_key);
        
        if (false === $output) {
            $current_url = remove_query_arg('paged');
            
            $output = '<form method="get" action="' . esc_url($current_url) . '" class="woocommerce-ordering custom-sort-form">';
            
            foreach ($_GET as $key => $value) {
                if ($key !== 'custom_sort' && $key !== 'paged') {
                    $output .= '<input type="hidden" name="' . esc_attr($key) . '" value="' . esc_attr($value) . '">';
                }
            }
            
            $output .= '<select name="custom_sort" class="custom-sort-dropdown" onchange="this.form.submit()">';
            $output .= '<option value="">' . esc_html__('مرتب‌سازی پیش‌فرض', 'custom-product-sorting') . '</option>';
            
            foreach ($saved_sorts as $key => $label) {
                $selected = ($current_sort === $key) ? 'selected="selected"' : '';
                $output .= sprintf(
                    '<option value="%s" %s>%s</option>',
                    esc_attr($key),
                    $selected,
                    esc_html($label)
                );
            }
            
            $output .= '</select></form>';
            
            wp_cache_set($cache_key, $output, '', 3600);
        }

        return $output;
    }

    public function customize_ordering_args($args) {
        if (!isset($_GET['custom_sort'])) {
            return $args;
        }

        $custom_sort = sanitize_text_field($_GET['custom_sort']);
        $saved_sorts = get_option($this->option_name, array());
        
        if (!array_key_exists($custom_sort, $saved_sorts)) {
            return $args;
        }

        $order = isset($_GET['order']) ? strtoupper($_GET['order']) : 'DESC';
        $order = in_array($order, array('ASC', 'DESC')) ? $order : 'DESC';

        $args['meta_key'] = '_' . $custom_sort;
        $args['orderby'] = 'meta_value_num';
        $args['order'] = $order;
        
        unset($args['orderby_primary']);
        unset($args['orderby_secondary']);
        
        return $args;
    }

    public function save_custom_sort() {
        check_ajax_referer('custom-sort-nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $sort_key = sanitize_key($_POST['sort_key']);
        $sort_label = sanitize_text_field($_POST['sort_label']);
        
        $saved_sorts = get_option($this->option_name, array());
        $saved_sorts[$sort_key] = $sort_label;
        
        update_option($this->option_name, $saved_sorts);
        
        wp_send_json_success();
    }

    public function delete_custom_sort() {
        $sort_key = sanitize_key($_POST['sort_key']);
        check_ajax_referer('delete_sort_' . $sort_key, 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $saved_sorts = get_option($this->option_name, array());
        
        if (isset($saved_sorts[$sort_key])) {
            // Delete all meta values for this sort option
            global $wpdb;
            $wpdb->delete(
                $wpdb->postmeta,
                array('meta_key' => '_' . $sort_key),
                array('%s')
            );
            
            // Remove the sort option
            unset($saved_sorts[$sort_key]);
            update_option($this->option_name, $saved_sorts);
            
            wp_send_json_success();
        } else {
            wp_send_json_error('Sort option not found');
        }
    }

    public function clear_sort_values() {
        $sort_key = sanitize_key($_POST['sort_key']);
        check_ajax_referer('clear_values_' . $sort_key, 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;
        
        try {
            // Get all products that have this meta key
            $meta_key = '_' . $sort_key;
            
            // Delete all meta values for this sort option
            $result = $wpdb->delete(
                $wpdb->postmeta,
                array('meta_key' => $meta_key),
                array('%s')
            );
            
            if ($result !== false) {
                clean_post_cache($wpdb->posts);
                wp_send_json_success(array(
                    'message' => sprintf('Successfully cleared %d values', $result)
                ));
            } else {
                wp_send_json_error('Database error while clearing values');
            }
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function add_sorting_to_default_dropdown($sortby) {
        $saved_sorts = get_option($this->option_name, array());
        
        foreach ($saved_sorts as $key => $label) {
            $sortby[$key] = esc_html($label);
        }
        
        return $sortby;
    }
}

// Initialize plugin using singleton pattern
add_action('plugins_loaded', function() {
    Custom_Product_Sorting::get_instance();
});