<?php
/**
 * Plugin Name: Persian Date Converter
 * Plugin URI: https://yourwebsite.com
 * Description: A WordPress plugin to convert Gregorian dates to Jalali and vice versa with month names displayed in dropdown.
 * Version: 2.3.2
 * Author: MUSTAFA NOORMOHAMMADI
 * Author URI: https://yourwebsite.com
 * Text Domain: persian-date-converter
 */

if (!defined('ABSPATH')) {
    exit;
}

// Existing conversion functions remain the same...
function gregorian_to_jalali($gy, $gm, $gd) {
    $g_d_m = [0, 31, (365 + (int)date('L', strtotime("$gy-03-01")) - 337), 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    $jy = ($gy <= 1600) ? 0 : 979;
    $gy -= ($gy <= 1600) ? 621 : 1600;
    $gy2 = ($gm > 2) ? ($gy + 1) : $gy;
    $days = (365 * $gy) + (int)(($gy2 + 3) / 4) - (int)(($gy2 + 99) / 100) + (int)(($gy2 + 399) / 400) - 80 + $gd;
    for ($i = 0; $i < $gm; ++$i) $days += $g_d_m[$i];
    $jy += 33 * (int)($days / 12053);
    $days %= 12053;
    $jy += 4 * (int)($days / 1461);
    $days %= 1461;
    if ($days > 365) {
        $jy += (int)(($days - 1) / 365);
        $days = ($days - 1) % 365;
    }
    $jm = ($days < 186) ? 1 + (int)($days / 31) : 7 + (int)(($days - 186) / 30);
    $jd = 1 + (($days < 186) ? ($days % 31) : (($days - 186) % 30));
    return [$jy, $jm, $jd];
}

function jalali_to_gregorian($jy, $jm, $jd) {
    $jy += 1595;
    $days = -355668 + (365 * $jy) + (int)($jy / 33) * 8 + (int)((($jy % 33) + 3) / 4) + $jd + (($jm < 7) ? ($jm - 1) * 31 : (($jm - 7) * 30) + 186);
    $gy = 400 * (int)($days / 146097);
    $days %= 146097;
    if ($days > 36524) {
        $gy += 100 * (int)(--$days / 36524);
        $days %= 36524;
        if ($days >= 365) $days++;
    }
    $gy += 4 * (int)($days / 1461);
    $days %= 1461;
    if ($days > 365) {
        $gy += (int)(($days - 1) / 365);
        $days = ($days - 1) % 365;
    }
    $gd = $days + 1;
    foreach ([0, 31, (365 + (int)date('L', strtotime("$gy-03-01")) - 337), 31, 30, 31, 30, 31, 31, 30, 31, 30, 31] as $gm => $v)
        if ($gd <= $v) break;
        else $gd -= $v;
    return [$gy, $gm, $gd];
}

// Existing AJAX handler and utility functions remain the same...

// Enqueue scripts with optimized JavaScript
add_action('wp_enqueue_scripts', function () {
    ?>
    <style>
    #date-converter {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        align-items: center;
    }
    #date-converter > * {
        flex: 1;
        min-width: 100px;
    }
    #result-container {
        width: 100%;
        background-color: rgba(128, 128, 128, 0.1);
        border-radius: 8px;
        padding: 10px;
        margin-top: 10px;
        opacity: 0;
        max-height: 0;
        overflow: hidden;
        transition: all 0.3s ease-out;
    }
    #result-container.show {
        opacity: 1;
        max-height: 200px;
    }
    #result-label {
        font-weight: bold;
        color: #555;
    }
    #result {
        color: #333;
        font-size: 1.1em;
    }
    </style>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const $ = {
            jalaliMonths: ['فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'],
            gregorianMonths: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'],
            
            els: {
                convertType: document.getElementById('convert-type'),
                monthDropdown: document.getElementById('month'),
                dayInput: document.getElementById('day'),
                yearInput: document.getElementById('year'),
                convertBtn: document.getElementById('convert-btn'),
                resultContainer: document.getElementById('result-container'),
                resultElement: document.getElementById('result')
            },

            init() {
                this.populateMonths();
                this.bindEvents();
            },

            populateMonths() {
                const months = this.els.convertType.value === 'gregorian_to_jalali' 
                    ? this.gregorianMonths 
                    : this.jalaliMonths;
                
                this.els.monthDropdown.innerHTML = months
                    .map(month => `<option value="${month}">${month}</option>`)
                    .join('');
            },

            bindEvents() {
                this.els.convertType.addEventListener('change', () => this.populateMonths());
                this.els.convertBtn.addEventListener('click', () => this.convertDate());
            },

            convertDate() {
                const { dayInput, monthDropdown, yearInput, convertType, resultContainer, resultElement } = this.els;
                
                if (!dayInput.value || !monthDropdown.value || !yearInput.value) {
                    alert("لطفاً تمام فیلدها را پر کنید.");
                    return;
                }

                resultContainer.classList.remove('show');

                const formData = new FormData();
                formData.append('action', 'date_conversion');
                formData.append('type', convertType.value);
                formData.append('day', dayInput.value);
                formData.append('month', monthDropdown.value);
                formData.append('year', yearInput.value);

                fetch("<?php echo admin_url('admin-ajax.php'); ?>", {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(result => {
                    if (result.success) {
                        resultElement.innerHTML = `
                            ${result.data.formatted} <br> 
                            ${result.data.formatted_text}
                        `;
                        // Minimal delay for smoother animation
                        requestAnimationFrame(() => {
                            resultContainer.classList.add('show');
                        });
                    } else {
                        resultElement.innerHTML = "خطا در تبدیل تاریخ: " + result.data;
                        resultContainer.classList.add('show');
                    }
                })
                .catch(error => {
                    resultElement.innerHTML = "خطا در برقراری ارتباط: " + error;
                    resultContainer.classList.add('show');
                });
            }
        };

        $.init();
    });
    </script>
    <?php
});

// Shortcode remains the same
add_shortcode('persian_date_converter', function () {
    return '
        <div id="date-converter">
            <select id="convert-type">
                <option value="gregorian_to_jalali">میلادی به شمسی</option>
                <option value="jalali_to_gregorian">شمسی به میلادی</option>
            </select>
            <input type="number" id="day" placeholder="روز" min="1" max="31">
            <select id="month" placeholder="ماه"></select>
            <input type="number" id="year" placeholder="سال" min="1">
            <button id="convert-btn">تبدیل</button>
            <div id="result-container">
                <span id="result-label">نتیجه تبدیل:</span>
                <div id="result"></div>
            </div>
        </div>
    ';
});