<?php
/*
Plugin Name: OSM Distance Calculator
Plugin URI: https://github.com/mostafanormohammadi/osm-distance-calculator
Description: محاسبه مسافت و مسیر با استفاده از نقشه OpenStreetMap
Version: 1.3
Author: مصطفی نورمحمدی
Author URI: https://t.me/mostafanormohammadi
License: GPL2
*/

// امنیت پلاگین
if (!defined('ABSPATH')) {
    exit;
}

class OSMDistanceCalculatorPlugin {
    public function __construct() {
        // ثبت اکشن‌ها و شورت کدها
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_shortcode('osm_distance_calculator', [$this, 'render_distance_calculator']);
        
        // افزودن صفحه تنظیمات
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    // بارگذاری اسکریپت‌ها و استایل‌ها
    public function enqueue_scripts() {
        // اسکریپت‌های نقشه
        wp_enqueue_script('leaflet-js', 'https://unpkg.com/leaflet@1.7.1/dist/leaflet.js', [], '1.7.1', true);
        wp_enqueue_script('leaflet-routing', 'https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.js', ['leaflet-js'], '3.2.12', true);
        
        // اسکریپت اختصاصی
        wp_enqueue_script('osm-distance-calculator', plugin_dir_url(__FILE__) . 'script.js', ['jquery', 'leaflet-js', 'leaflet-routing'], '1.3', true);
        
        // انتقال تنظیمات به اسکریپت
        $osrm_server = get_option('osm_distance_calculator_osrm_server', 'https://router.project-osrm.org');
        wp_localize_script('osm-distance-calculator', 'OSMCalculatorSettings', [
            'osrmServer' => $osrm_server
        ]);
        
        // استایل‌ها
        wp_enqueue_style('leaflet-css', 'https://unpkg.com/leaflet@1.7.1/dist/leaflet.css');
        wp_enqueue_style('leaflet-routing-css', 'https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.css');
        wp_enqueue_style('osm-distance-calculator', plugin_dir_url(__FILE__) . 'style.css');
    }

    // صفحه تنظیمات پلاگین
    public function add_settings_page() {
        add_options_page(
            'تنظیمات محاسبه گر مسافت', 
            'محاسبه گر مسافت', 
            'manage_options', 
            'osm-distance-calculator', 
            [$this, 'render_settings_page']
        );
    }

    // ثبت تنظیمات
    public function register_settings() {
        register_setting('osm_distance_calculator_options', 'osm_distance_calculator_osrm_server');
    }

    // رندر صفحه تنظیمات
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>تنظیمات محاسبه گر مسافت</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('osm_distance_calculator_options');
                do_settings_sections('osm_distance_calculator_options');
                ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">سرور OSRM</th>
                        <td>
                            <input type="text" name="osm_distance_calculator_osrm_server" 
                                   value="<?php echo esc_attr(get_option('osm_distance_calculator_osrm_server', 'https://router.project-osrm.org')); ?>" 
                                   class="regular-text"/>
                            <p class="description">آدرس سرور مسیریابی OSRM را وارد کنید</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    // رندر محاسبه گر مسافت
    public function render_distance_calculator() {
        ob_start();
        ?>
        <div class="osm-distance-calculator-container">
            <div class="map-container">
                <div id="osm-distance-map"></div>
            </div>
            <div class="form-container">
                <form id="distance-form">
                    <div class="form-group">
                        <label>مبدأ</label>
                        <input type="text" id="origin-input" class="location-input" placeholder="نام شهر یا آدرس مبدأ">
                    </div>
                    <div class="form-group">
                        <label>مقصد</label>
                        <input type="text" id="destination-input" class="location-input" placeholder="نام شهر یا آدرس مقصد">
                    </div>
                    <button type="button" id="calculate-distance">محاسبه فاصله</button>
                    <div id="result-container">
                        <p>فاصله: <span id="distance-result">-</span></p>
                        <p>زمان تقریبی: <span id="duration-result">-</span></p>
                    </div>
                </form>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}

// اسکریپت JavaScript در داخل همین فایل
function osm_distance_calculator_inline_script() {
    ?>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        let map, originMarker, destinationMarker, routeLine;

        function initMap() {
            map = L.map('osm-distance-map').setView([32.4279, 53.6880], 5);

            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
            }).addTo(map);

            document.getElementById('calculate-distance').addEventListener('click', calculateDistance);
        }

        function addMarkerWithLabel(location, type) {
            const color = type === 'origin' ? 'blue' : 'red';
            const marker = L.marker(location, {
                icon: L.divIcon({
                    className: 'custom-marker',
                    html: `<div class="marker-label" style="background-color:${color};">${
                        type === 'origin' ? 'مبدأ' : 'مقصد'
                    }</div>`,
                    iconSize: [100, 40],
                    iconAnchor: [50, 0]
                })
            }).addTo(map);

            return marker;
        }

        function calculateDistance() {
            const originInput = document.getElementById('origin-input');
            const destinationInput = document.getElementById('destination-input');

            // حذف نشانگرها و مسیر قبلی
            if (originMarker) map.removeLayer(originMarker);
            if (destinationMarker) map.removeLayer(destinationMarker);
            if (routeLine) map.removeLayer(routeLine);

            // محاسبه موقعیت مکانی با Nominatim
            fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(originInput.value)}`)
                .then(response => response.json())
                .then(originResults => {
                    if (originResults.length > 0) {
                        const originLocation = [
                            parseFloat(originResults[0].lon), 
                            parseFloat(originResults[0].lat)
                        ];
                        originMarker = addMarkerWithLabel([originLocation[1], originLocation[0]], 'origin');

                        return fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(destinationInput.value)}`)
                            .then(response => response.json())
                            .then(destResults => {
                                if (destResults.length > 0) {
                                    const destLocation = [
                                        parseFloat(destResults[0].lon), 
                                        parseFloat(destResults[0].lat)
                                    ];
                                    destinationMarker = addMarkerWithLabel([destLocation[1], destLocation[0]], 'destination');

                                    // محاسبه مسیر با استفاده از OSRM
                                    const osrmServer = OSMCalculatorSettings.osrmServer || 'https://router.project-osrm.org';
                                    fetch(`${osrmServer}/route/v1/driving/${originLocation[0]},${originLocation[1]};${destLocation[0]},${destLocation[1]}?overview=full&geometries=geojson`)
                                        .then(response => response.json())
                                        .then(routeData => {
                                            if (routeData.routes && routeData.routes.length > 0) {
                                                const route = routeData.routes[0];
                                                const distance = route.distance / 1000; // تبدیل متر به کیلومتر
                                                const duration = route.duration / 60; // تبدیل ثانیه به دقیقه

                                                // نمایش نتایج
                                                document.getElementById('distance-result').textContent = `${distance.toFixed(1)} کیلومتر`;
                                                document.getElementById('duration-result').textContent = formatDuration(duration);

                                                // رسم مسیر روی نقشه
                                                const routeCoordinates = route.geometry.coordinates.map(coord => [coord[1], coord[0]]);
                                                routeLine = L.polyline(routeCoordinates, {
                                                    color: 'purple',
                                                    weight: 5,
                                                    opacity: 0.7
                                                }).addTo(map);

                                                // تنظیم نمای نقشه برای نمایش کل مسیر
                                                map.fitBounds(routeLine.getBounds());
                                            }
                                        })
                                        .catch(error => {
                                            console.error('خطا در محاسبه مسیر:', error);
                                            alert('خطا در محاسبه مسیر. لطفاً دوباره امتحان کنید.');
                                        });
                                }
                            });
                    }
                })
                .catch(error => {
                    console.error('خطا در محاسبه مسافت:', error);
                    alert('خطا در محاسبه فاصله. لطفاً نام شهر را به درستی وارد کنید.');
                });
        }

        // تابع فرمت‌بندی زمان
        function formatDuration(minutes) {
            const hours = Math.floor(minutes / 60);
            const remainingMinutes = Math.round(minutes % 60);
            
            if (hours > 0) {
                return `${hours} ساعت و ${remainingMinutes} دقیقه`;
            }
            return `${Math.round(minutes)} دقیقه`;
        }

        // تنظیم نقشه پس از بارگذاری
        window.addEventListener('load', initMap);
    });
    </script>
    <?php
}
add_action('wp_footer', 'osm_distance_calculator_inline_script');

// اسکریپت استایل CSS
function osm_distance_calculator_inline_style() {
    ?>
    <style>
    .osm-distance-calculator-container {
        display: flex;
        direction: rtl;
        max-width: 1200px;
        margin: 0 auto;
    }

    .map-container {
        width: 60%;
        height: 500px;
    }

    #osm-distance-map {
        width: 100%;
        height: 100%;
    }

    .form-container {
        width: 40%;
        padding: 20px;
        background-color: #f4f4f4;
    }

    .form-group {
        margin-bottom: 15px;
    }

    .location-input {
        width: 100%;
        padding: 10px;
        margin-top: 5px;
    }

    #calculate-distance {
        width: 100%;
        padding: 10px;
        background-color: #007bff;
        color: white;
        border: none;
        cursor: pointer;
    }

    #result-container {
        margin-top: 20px;
        text-align: center;
    }

    .custom-marker .marker-label {
        padding: 5px 10px;
        color: white;
        border-radius: 5px;
        font-size: 12px;
        display: inline-block;
        text-align: center;
    }
    </style>
    <?php
}
add_action('wp_head', 'osm_distance_calculator_inline_style');

// راه‌اندازی پلاگین
function initialize_osm_distance_calculator() {
    new OSMDistanceCalculatorPlugin();
}
add_action('plugins_loaded', 'initialize_osm_distance_calculator');